"""
Akemi Cost Suite - M0: Configuracion General
Datos empresa, parametros legales, tasas BCV, respaldos
"""
import customtkinter as ctk
from utils.database import (get_connection, get_empresa, update_empresa,
                            get_param, backup_db)
from utils.common import *


class M0Config(ctk.CTkFrame):
    def __init__(self, master, status_bar=None):
        super().__init__(master, fg_color="transparent")
        self.status = status_bar
        self._build_ui()

    def _build_ui(self):
        # Tabs internos
        self.tabs = ctk.CTkTabview(self, anchor="nw")
        self.tabs.pack(fill="both", expand=True, padx=5, pady=5)

        self.tabs.add("Empresa")
        self.tabs.add("Parametros Legales")
        self.tabs.add("Tasa BCV")
        self.tabs.add("Periodos")
        self.tabs.add("Respaldos")

        self._build_empresa_tab()
        self._build_params_tab()
        self._build_bcv_tab()
        self._build_periodos_tab()
        self._build_backup_tab()

    # === TAB EMPRESA ===
    def _build_empresa_tab(self):
        tab = self.tabs.tab("Empresa")
        frame = ctk.CTkScrollableFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=10)

        ctk.CTkLabel(frame, text="Datos de la Empresa",
                     font=ctk.CTkFont(size=18, weight="bold"),
                     text_color=NAVY).pack(anchor="w", pady=(0, 15))

        self.emp_fields = {}
        fields = [
            ("rif", "RIF", "J-12345678-9"),
            ("razon_social", "Razon Social", "Nombre de la empresa"),
            ("direccion", "Direccion", "Direccion completa"),
            ("telefono", "Telefono", "0412-1234567"),
            ("actividad_economica", "Actividad Economica", "Manufactura, Comercio, etc."),
            ("clasificacion_sundde", "Clasificacion SUNDDE", "Productor, Importador, etc."),
        ]

        for key, label, ph in fields:
            le = LabelEntry(frame, label, width=350, placeholder=ph)
            le.pack(fill="x", pady=4)
            self.emp_fields[key] = le

        # Horas semanales
        self.hrs_entry = LabelEntry(frame, "Horas Semanales", width=100, placeholder="44")
        self.hrs_entry.pack(fill="x", pady=4)

        # Moneda base
        self.moneda_combo = LabelCombo(frame, "Moneda Base", ["Bs", "USD"])
        self.moneda_combo.pack(fill="x", pady=4)

        # Botones
        btn_frame = ctk.CTkFrame(frame, fg_color="transparent")
        btn_frame.pack(fill="x", pady=20)
        ctk.CTkButton(btn_frame, text="Guardar Datos", fg_color=GREEN,
                      width=160, command=self._save_empresa).pack(side="left", padx=5)

        self._load_empresa()

    def _load_empresa(self):
        emp = get_empresa()
        if emp:
            for key, le in self.emp_fields.items():
                le.set(emp.get(key, ""))
            self.hrs_entry.set(emp.get("horas_semanales", 44))
            self.moneda_combo.set(emp.get("moneda_base", "Bs"))

    def _save_empresa(self):
        data = {k: le.get() for k, le in self.emp_fields.items()}
        try:
            data["horas_semanales"] = float(self.hrs_entry.get() or 44)
        except ValueError:
            data["horas_semanales"] = 44
        data["moneda_base"] = self.moneda_combo.get()
        update_empresa(**data)
        if self.status:
            self.status.set_status("Datos de empresa guardados correctamente")
        show_message(self, "Guardado", "Datos de empresa actualizados.", "success")

    # === TAB PARAMETROS LEGALES ===
    def _build_params_tab(self):
        tab = self.tabs.tab("Parametros Legales")
        frame = ctk.CTkScrollableFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=10)

        ctk.CTkLabel(frame, text="Parametros Legales Editables",
                     font=ctk.CTkFont(size=18, weight="bold"),
                     text_color=NAVY).pack(anchor="w", pady=(0, 5))
        ctk.CTkLabel(frame, text="Estos valores se usan para calcular cargas patronales, "
                     "salario integral y estructura SUNDDE.",
                     font=ctk.CTkFont(size=12),
                     text_color=GRAY).pack(anchor="w", pady=(0, 15))

        # Header
        hdr = ctk.CTkFrame(frame, fg_color=NAVY, corner_radius=4)
        hdr.pack(fill="x", pady=(0, 2))
        for txt, w in [("Parametro", 280), ("Valor %", 80), ("Base Legal", 200)]:
            ctk.CTkLabel(hdr, text=txt, width=w, text_color="white",
                         font=ctk.CTkFont(size=12, weight="bold")).pack(side="left", padx=4, pady=4)

        self.param_entries = {}
        conn = get_connection()
        rows = conn.execute(
            "SELECT * FROM parametros_legales ORDER BY id"
        ).fetchall()
        conn.close()

        for i, row in enumerate(rows):
            bg = VERY_LIGHT if i % 2 == 0 else WHITE
            rf = ctk.CTkFrame(frame, fg_color=bg, corner_radius=2)
            rf.pack(fill="x", pady=1)

            ctk.CTkLabel(rf, text=row["descripcion"], width=280,
                         font=ctk.CTkFont(size=12), anchor="w").pack(side="left", padx=4, pady=3)

            entry = ctk.CTkEntry(rf, width=80, font=ctk.CTkFont(size=12))
            entry.insert(0, str(row["valor"]))
            entry.pack(side="left", padx=4)
            self.param_entries[row["clave"]] = entry

            ctk.CTkLabel(rf, text=row["base_legal"], width=200,
                         font=ctk.CTkFont(size=11), text_color=GRAY,
                         anchor="w").pack(side="left", padx=4)

        ctk.CTkButton(frame, text="Guardar Parametros", fg_color=GREEN,
                      width=180, command=self._save_params).pack(pady=15, anchor="w")

    def _save_params(self):
        conn = get_connection()
        for clave, entry in self.param_entries.items():
            try:
                val = float(entry.get())
            except ValueError:
                continue
            conn.execute(
                "UPDATE parametros_legales SET valor=? WHERE clave=?", (val, clave)
            )
        conn.commit()
        conn.close()
        if self.status:
            self.status.set_status("Parametros legales actualizados")
        show_message(self, "Guardado", "Parametros legales actualizados.", "success")

    # === TAB TASA BCV ===
    def _build_bcv_tab(self):
        tab = self.tabs.tab("Tasa BCV")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=10)

        ctk.CTkLabel(frame, text="Tasa de Cambio BCV (Bs/USD)",
                     font=ctk.CTkFont(size=18, weight="bold"),
                     text_color=NAVY).pack(anchor="w", pady=(0, 15))

        input_frame = ctk.CTkFrame(frame, fg_color="transparent")
        input_frame.pack(fill="x", pady=5)

        self.bcv_fecha = LabelEntry(input_frame, "Fecha", width=130,
                                    placeholder=fecha_hoy())
        self.bcv_fecha.pack(side="left", padx=(0, 10))
        self.bcv_fecha.set(fecha_hoy())

        self.bcv_tasa = LabelEntry(input_frame, "Tasa USD", width=130,
                                   placeholder="0.00")
        self.bcv_tasa.pack(side="left", padx=(0, 10))

        ctk.CTkButton(input_frame, text="Registrar", fg_color=GREEN,
                      width=100, command=self._save_bcv).pack(side="left")

        # Historial
        ctk.CTkLabel(frame, text="Historial de Tasas",
                     font=ctk.CTkFont(size=14, weight="bold"),
                     text_color=BLUE).pack(anchor="w", pady=(20, 5))

        self.bcv_table = ScrollableTable(
            frame, columns=["Fecha", "Tasa Bs/USD", "Fuente"],
            widths=[150, 150, 120], height=250
        )
        self.bcv_table.pack(fill="both", expand=True)
        self._load_bcv()

    def _load_bcv(self):
        self.bcv_table.clear_rows()
        conn = get_connection()
        rows = conn.execute(
            "SELECT * FROM tasas_bcv ORDER BY fecha DESC LIMIT 50"
        ).fetchall()
        conn.close()
        for r in rows:
            self.bcv_table.add_row([
                fmt_fecha(r["fecha"]),
                fmt_num(r["tasa_usd"], 4),
                r["fuente"]
            ])

    def _save_bcv(self):
        fecha = self.bcv_fecha.get()
        try:
            tasa = float(self.bcv_tasa.get())
        except (ValueError, TypeError):
            show_message(self, "Error", "Ingrese una tasa valida.", "error")
            return
        if tasa <= 0:
            show_message(self, "Error", "La tasa debe ser mayor a 0.", "error")
            return
        conn = get_connection()
        conn.execute(
            "INSERT INTO tasas_bcv (fecha, tasa_usd, fuente) VALUES (?,?,?)",
            (fecha, tasa, "manual")
        )
        conn.commit()
        conn.close()
        self.bcv_tasa.set("")
        self._load_bcv()
        if self.status:
            self.status.set_status(f"Tasa BCV {tasa} registrada para {fecha}")

    # === TAB PERIODOS ===
    def _build_periodos_tab(self):
        tab = self.tabs.tab("Periodos")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=10)

        ctk.CTkLabel(frame, text="Periodos Contables",
                     font=ctk.CTkFont(size=18, weight="bold"),
                     text_color=NAVY).pack(anchor="w", pady=(0, 15))

        # Crear periodo
        add_frame = ctk.CTkFrame(frame, fg_color="transparent")
        add_frame.pack(fill="x", pady=5)

        self.per_anio = LabelEntry(add_frame, "Ano", width=80, placeholder="2026")
        self.per_anio.pack(side="left", padx=(0, 10))

        self.per_mes = LabelCombo(add_frame, "Mes",
                                  [str(i) for i in range(1, 13)], width=80)
        self.per_mes.pack(side="left", padx=(0, 10))

        ctk.CTkButton(add_frame, text="Crear Periodo", fg_color=GREEN,
                      width=120, command=self._add_periodo).pack(side="left")

        self.per_table = ScrollableTable(
            frame, columns=["Ano", "Mes", "Estado"],
            widths=[120, 80, 120], height=250
        )
        self.per_table.pack(fill="both", expand=True, pady=10)
        self._load_periodos()

    def _load_periodos(self):
        self.per_table.clear_rows()
        conn = get_connection()
        rows = conn.execute(
            "SELECT * FROM periodos ORDER BY anio DESC, mes DESC LIMIT 24"
        ).fetchall()
        conn.close()
        for r in rows:
            self.per_table.add_row([r["anio"], r["mes"], r["estado"]])

    def _add_periodo(self):
        try:
            anio = int(self.per_anio.get())
            mes = int(self.per_mes.get())
        except (ValueError, TypeError):
            show_message(self, "Error", "Ingrese ano y mes validos.", "error")
            return
        conn = get_connection()
        try:
            conn.execute(
                "INSERT INTO periodos (anio, mes) VALUES (?,?)", (anio, mes)
            )
            conn.commit()
            self._load_periodos()
            if self.status:
                self.status.set_status(f"Periodo {mes}/{anio} creado")
        except Exception:
            show_message(self, "Error", "Ese periodo ya existe.", "error")
        finally:
            conn.close()

    # === TAB RESPALDOS ===
    def _build_backup_tab(self):
        tab = self.tabs.tab("Respaldos")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=10)

        ctk.CTkLabel(frame, text="Respaldo de Base de Datos",
                     font=ctk.CTkFont(size=18, weight="bold"),
                     text_color=NAVY).pack(anchor="w", pady=(0, 15))

        ctk.CTkLabel(frame, text="Crear una copia de seguridad de todos los datos del sistema.",
                     font=ctk.CTkFont(size=13)).pack(anchor="w", pady=(0, 20))

        ctk.CTkButton(frame, text="Crear Respaldo Ahora", fg_color=GREEN,
                      width=200, height=40,
                      font=ctk.CTkFont(size=14),
                      command=self._do_backup).pack(anchor="w")

        self.backup_label = ctk.CTkLabel(frame, text="", font=ctk.CTkFont(size=12),
                                         text_color=TEAL)
        self.backup_label.pack(anchor="w", pady=10)

    def _do_backup(self):
        path = backup_db()
        if path:
            self.backup_label.configure(text=f"Respaldo creado: {path}")
            if self.status:
                self.status.set_status("Respaldo creado exitosamente")
            show_message(self, "Respaldo", f"Archivo guardado en:\n{path}", "success")
        else:
            show_message(self, "Error", "No se pudo crear el respaldo.", "error")
