"""
Akemi Cost Suite - M1: Materia Prima e Inventarios
Catalogo, entradas, salidas, kardex PEPS/CPP
"""
import customtkinter as ctk
from utils.database import get_connection, get_param
from utils.common import *


class M1Inventarios(ctk.CTkFrame):
    def __init__(self, master, status_bar=None):
        super().__init__(master, fg_color="transparent")
        self.status = status_bar
        self._build_ui()

    def _build_ui(self):
        self.tabs = ctk.CTkTabview(self, anchor="nw")
        self.tabs.pack(fill="both", expand=True, padx=5, pady=5)

        self.tabs.add("Catalogo Materiales")
        self.tabs.add("Entradas (Compras)")
        self.tabs.add("Salidas (Requisiciones)")
        self.tabs.add("Kardex")

        self._build_catalogo()
        self._build_entradas()
        self._build_salidas()
        self._build_kardex()

    # ==========================================
    # TAB: CATALOGO DE MATERIALES
    # ==========================================
    def _build_catalogo(self):
        tab = self.tabs.tab("Catalogo Materiales")

        # Panel superior: formulario
        top = ctk.CTkFrame(tab, fg_color="transparent")
        top.pack(fill="x", padx=10, pady=5)

        ctk.CTkLabel(top, text="Catalogo de Materiales",
                     font=ctk.CTkFont(size=18, weight="bold"),
                     text_color=NAVY).pack(anchor="w", pady=(0, 10))

        form = ctk.CTkFrame(top, fg_color="transparent")
        form.pack(fill="x")

        left = ctk.CTkFrame(form, fg_color="transparent")
        left.pack(side="left", fill="both", expand=True)
        right = ctk.CTkFrame(form, fg_color="transparent")
        right.pack(side="left", fill="both", expand=True, padx=(15, 0))

        self.mat_codigo = LabelEntry(left, "Codigo", placeholder="MP-001")
        self.mat_codigo.pack(fill="x", pady=3)
        self.mat_desc = LabelEntry(left, "Descripcion", placeholder="Nombre del material")
        self.mat_desc.pack(fill="x", pady=3)
        self.mat_unidad = LabelCombo(left, "Unidad Medida",
                                     ["Und", "Kg", "Lts", "Galon", "Metros", "Lamina",
                                      "Resma", "Rollo", "Caja", "Paquete"])
        self.mat_unidad.pack(fill="x", pady=3)
        self.mat_tipo = LabelCombo(left, "Tipo",
                                   ["directo", "indirecto"])
        self.mat_tipo.pack(fill="x", pady=3)

        # Categorias
        conn = get_connection()
        cats = [r["nombre"] for r in conn.execute(
            "SELECT nombre FROM categorias_material ORDER BY nombre").fetchall()]
        conn.close()
        self.mat_cat = LabelCombo(right, "Categoria", cats if cats else ["General"])
        self.mat_cat.pack(fill="x", pady=3)

        self.mat_proveedor = LabelEntry(right, "Proveedor", placeholder="Nombre proveedor")
        self.mat_proveedor.pack(fill="x", pady=3)
        self.mat_precio_bs = LabelEntry(right, "Precio Unit. Bs", placeholder="0.00")
        self.mat_precio_bs.pack(fill="x", pady=3)
        self.mat_stock_min = LabelEntry(right, "Stock Minimo", placeholder="0")
        self.mat_stock_min.pack(fill="x", pady=3)

        # Presentacion comercial
        pres_frame = ctk.CTkFrame(top, fg_color="transparent")
        pres_frame.pack(fill="x", pady=5)
        self.mat_presentacion = LabelEntry(pres_frame, "Presentacion", width=180,
                                           placeholder="Ej: Resma 500 hojas")
        self.mat_presentacion.pack(side="left", padx=(0, 10))
        self.mat_cant_pres = LabelEntry(pres_frame, "Cant/Pres.", width=80,
                                        placeholder="500")
        self.mat_cant_pres.pack(side="left", padx=(0, 10))
        self.mat_precio_pres = LabelEntry(pres_frame, "Precio Pres.", width=100,
                                          placeholder="58.00")
        self.mat_precio_pres.pack(side="left", padx=(0, 10))
        ctk.CTkButton(pres_frame, text="Calc. P.Unit.", width=100,
                      fg_color=TEAL, command=self._calc_precio_unit).pack(side="left")

        # Botones
        btn_f = ctk.CTkFrame(top, fg_color="transparent")
        btn_f.pack(fill="x", pady=8)
        ctk.CTkButton(btn_f, text="Guardar Material", fg_color=GREEN,
                      width=150, command=self._save_material).pack(side="left", padx=5)
        ctk.CTkButton(btn_f, text="Limpiar", fg_color=GRAY,
                      width=100, command=self._clear_mat_form).pack(side="left", padx=5)
        ctk.CTkButton(btn_f, text="Eliminar", fg_color=RED,
                      width=100, command=self._delete_material).pack(side="left", padx=5)

        self.mat_edit_id = None

        # Tabla
        self.mat_table = ScrollableTable(
            tab,
            columns=["ID", "Codigo", "Descripcion", "Unidad", "Tipo", "Precio Bs", "Stock"],
            widths=[40, 80, 180, 60, 70, 100, 70],
            height=250
        )
        self.mat_table.pack(fill="both", expand=True, padx=10, pady=5)
        self._load_materiales()

    def _calc_precio_unit(self):
        try:
            cant = float(self.mat_cant_pres.get() or 1)
            precio = float(self.mat_precio_pres.get() or 0)
            if cant > 0:
                pu = precio / cant
                self.mat_precio_bs.set(f"{pu:.4f}")
        except ValueError:
            pass

    def _load_materiales(self):
        self.mat_table.clear_rows()
        conn = get_connection()
        rows = conn.execute(
            "SELECT * FROM materiales WHERE activo=1 ORDER BY codigo"
        ).fetchall()
        conn.close()
        for r in rows:
            self.mat_table.add_row(
                [r["id"], r["codigo"], r["descripcion"], r["unidad_medida"],
                 r["tipo"], fmt_num(r["precio_unitario_bs"], 4), fmt_num(r["stock_actual"])],
                tag=r["id"],
                on_click=self._select_material
            )

    def _select_material(self, mat_id):
        conn = get_connection()
        r = conn.execute("SELECT * FROM materiales WHERE id=?", (mat_id,)).fetchone()
        conn.close()
        if not r:
            return
        self.mat_edit_id = r["id"]
        self.mat_codigo.set(r["codigo"])
        self.mat_desc.set(r["descripcion"])
        self.mat_unidad.set(r["unidad_medida"])
        self.mat_tipo.set(r["tipo"])
        self.mat_proveedor.set(r["proveedor"])
        self.mat_precio_bs.set(r["precio_unitario_bs"])
        self.mat_stock_min.set(r["stock_minimo"])
        self.mat_presentacion.set(r["presentacion_comercial"])
        self.mat_cant_pres.set(r["cantidad_por_presentacion"])

        # Buscar categoria
        if r["categoria_id"]:
            conn2 = get_connection()
            cat = conn2.execute(
                "SELECT nombre FROM categorias_material WHERE id=?",
                (r["categoria_id"],)
            ).fetchone()
            conn2.close()
            if cat:
                self.mat_cat.set(cat["nombre"])

    def _save_material(self):
        codigo = self.mat_codigo.get()
        desc = self.mat_desc.get()
        if not codigo or not desc:
            show_message(self, "Error", "Codigo y Descripcion son obligatorios.", "error")
            return

        unidad = self.mat_unidad.get()
        tipo = self.mat_tipo.get()
        prov = self.mat_proveedor.get()
        try:
            precio = float(self.mat_precio_bs.get() or 0)
        except ValueError:
            precio = 0
        try:
            stock_min = float(self.mat_stock_min.get() or 0)
        except ValueError:
            stock_min = 0
        pres = self.mat_presentacion.get()
        try:
            cant_pres = float(self.mat_cant_pres.get() or 1)
        except ValueError:
            cant_pres = 1

        # Obtener categoria_id
        cat_nombre = self.mat_cat.get()
        conn = get_connection()
        cat_row = conn.execute(
            "SELECT id FROM categorias_material WHERE nombre=?", (cat_nombre,)
        ).fetchone()
        cat_id = cat_row["id"] if cat_row else None

        try:
            if self.mat_edit_id:
                conn.execute("""UPDATE materiales SET
                    codigo=?, descripcion=?, unidad_medida=?, tipo=?, categoria_id=?,
                    proveedor=?, precio_unitario_bs=?, stock_minimo=?,
                    presentacion_comercial=?, cantidad_por_presentacion=?,
                    updated_at=datetime('now','localtime')
                    WHERE id=?""",
                    (codigo, desc, unidad, tipo, cat_id, prov, precio, stock_min,
                     pres, cant_pres, self.mat_edit_id))
                msg = "Material actualizado"
            else:
                conn.execute("""INSERT INTO materiales
                    (codigo, descripcion, unidad_medida, tipo, categoria_id,
                     proveedor, precio_unitario_bs, stock_minimo,
                     presentacion_comercial, cantidad_por_presentacion)
                    VALUES (?,?,?,?,?,?,?,?,?,?)""",
                    (codigo, desc, unidad, tipo, cat_id, prov, precio, stock_min,
                     pres, cant_pres))
                msg = "Material registrado"
            conn.commit()
            self._clear_mat_form()
            self._load_materiales()
            if self.status:
                self.status.set_status(msg)
        except Exception as e:
            show_message(self, "Error", f"Error: {str(e)}", "error")
        finally:
            conn.close()

    def _delete_material(self):
        if not self.mat_edit_id:
            return
        if confirm_dialog(self, "Confirmar", "Desea desactivar este material?"):
            conn = get_connection()
            conn.execute("UPDATE materiales SET activo=0 WHERE id=?", (self.mat_edit_id,))
            conn.commit()
            conn.close()
            self._clear_mat_form()
            self._load_materiales()

    def _clear_mat_form(self):
        self.mat_edit_id = None
        for w in [self.mat_codigo, self.mat_desc, self.mat_proveedor,
                  self.mat_precio_bs, self.mat_stock_min,
                  self.mat_presentacion, self.mat_cant_pres, self.mat_precio_pres]:
            w.set("")

    # ==========================================
    # TAB: ENTRADAS (COMPRAS)
    # ==========================================
    def _build_entradas(self):
        tab = self.tabs.tab("Entradas (Compras)")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)

        ctk.CTkLabel(frame, text="Registrar Entrada de Material (Compra)",
                     font=ctk.CTkFont(size=18, weight="bold"),
                     text_color=NAVY).pack(anchor="w", pady=(0, 10))

        form = ctk.CTkFrame(frame, fg_color="transparent")
        form.pack(fill="x")

        self.ent_material = LabelCombo(form, "Material", self._get_mat_list())
        self.ent_material.pack(fill="x", pady=3)
        self.ent_cantidad = LabelEntry(form, "Cantidad", placeholder="0")
        self.ent_cantidad.pack(fill="x", pady=3)
        self.ent_costo = LabelEntry(form, "Costo Unitario Bs", placeholder="0.00")
        self.ent_costo.pack(fill="x", pady=3)
        self.ent_factura = LabelEntry(form, "No. Factura", placeholder="F-0001")
        self.ent_factura.pack(fill="x", pady=3)
        self.ent_proveedor = LabelEntry(form, "Proveedor", placeholder="")
        self.ent_proveedor.pack(fill="x", pady=3)
        self.ent_fecha = LabelEntry(form, "Fecha", placeholder=fecha_hoy())
        self.ent_fecha.set(fecha_hoy())
        self.ent_fecha.pack(fill="x", pady=3)
        self.ent_lote = LabelEntry(form, "Lote", placeholder="Opcional")
        self.ent_lote.pack(fill="x", pady=3)

        ctk.CTkButton(form, text="Registrar Entrada", fg_color=GREEN,
                      width=180, command=self._save_entrada).pack(pady=10, anchor="w")

        # Ultimos movimientos
        ctk.CTkLabel(frame, text="Ultimas Entradas",
                     font=ctk.CTkFont(size=14, weight="bold"),
                     text_color=BLUE).pack(anchor="w", pady=(10, 5))

        self.ent_table = ScrollableTable(
            frame,
            columns=["Fecha", "Material", "Cant.", "C.Unit.", "Total", "Factura"],
            widths=[80, 150, 60, 80, 90, 80],
            height=180
        )
        self.ent_table.pack(fill="both", expand=True)
        self._load_entradas()

    def _get_mat_list(self):
        conn = get_connection()
        rows = conn.execute(
            "SELECT codigo || ' - ' || descripcion as txt FROM materiales WHERE activo=1 ORDER BY codigo"
        ).fetchall()
        conn.close()
        return [r["txt"] for r in rows] if rows else ["(sin materiales)"]

    def _refresh_mat_combos(self):
        mats = self._get_mat_list()
        self.ent_material.combo.configure(values=mats)
        self.sal_material.combo.configure(values=mats)
        self.kardex_material.combo.configure(values=mats)

    def _get_mat_id_from_combo(self, combo_val):
        if not combo_val or combo_val == "(sin materiales)":
            return None
        codigo = combo_val.split(" - ")[0].strip()
        conn = get_connection()
        row = conn.execute("SELECT id FROM materiales WHERE codigo=?", (codigo,)).fetchone()
        conn.close()
        return row["id"] if row else None

    def _save_entrada(self):
        mat_id = self._get_mat_id_from_combo(self.ent_material.get())
        if not mat_id:
            show_message(self, "Error", "Seleccione un material valido.", "error")
            return
        try:
            cant = float(self.ent_cantidad.get() or 0)
            costo = float(self.ent_costo.get() or 0)
        except ValueError:
            show_message(self, "Error", "Cantidad y costo deben ser numericos.", "error")
            return
        if cant <= 0:
            show_message(self, "Error", "La cantidad debe ser mayor a 0.", "error")
            return

        total = cant * costo
        fecha = self.ent_fecha.get() or fecha_hoy()
        factura = self.ent_factura.get()
        prov = self.ent_proveedor.get()
        lote = self.ent_lote.get()

        conn = get_connection()
        cur = conn.execute("""INSERT INTO movimientos_inventario
            (material_id, tipo, cantidad, costo_unitario, costo_total,
             factura, proveedor, fecha, lote)
            VALUES (?,?,?,?,?,?,?,?,?)""",
            (mat_id, "entrada", cant, costo, total, factura, prov, fecha, lote))
        mov_id = cur.lastrowid

        # Actualizar stock
        conn.execute(
            "UPDATE materiales SET stock_actual = stock_actual + ?, precio_unitario_bs=? WHERE id=?",
            (cant, costo, mat_id))

        # Registrar en kardex
        self._registrar_kardex_entrada(conn, mat_id, mov_id, fecha, cant, costo, total)

        conn.commit()
        conn.close()

        self.ent_cantidad.set("")
        self.ent_costo.set("")
        self.ent_factura.set("")
        self.ent_lote.set("")
        self._load_entradas()
        if self.status:
            self.status.set_status(f"Entrada registrada: {cant} unidades")

    def _registrar_kardex_entrada(self, conn, mat_id, mov_id, fecha, cant, costo, total):
        # Obtener ultimo saldo
        last = conn.execute(
            "SELECT cantidad_saldo, total_saldo FROM kardex WHERE material_id=? ORDER BY id DESC LIMIT 1",
            (mat_id,)
        ).fetchone()

        saldo_cant = (last["cantidad_saldo"] if last else 0) + cant
        saldo_total = (last["total_saldo"] if last else 0) + total
        saldo_unit = saldo_total / saldo_cant if saldo_cant > 0 else 0

        conn.execute("""INSERT INTO kardex
            (material_id, movimiento_id, fecha, tipo,
             cantidad_entrada, costo_unit_entrada, total_entrada,
             cantidad_saldo, costo_unit_saldo, total_saldo)
            VALUES (?,?,?,?,?,?,?,?,?,?)""",
            (mat_id, mov_id, fecha, "entrada",
             cant, costo, total,
             saldo_cant, saldo_unit, saldo_total))

    def _load_entradas(self):
        self.ent_table.clear_rows()
        conn = get_connection()
        rows = conn.execute("""
            SELECT m.fecha, mat.descripcion, m.cantidad, m.costo_unitario,
                   m.costo_total, m.factura
            FROM movimientos_inventario m
            JOIN materiales mat ON m.material_id = mat.id
            WHERE m.tipo='entrada'
            ORDER BY m.id DESC LIMIT 30
        """).fetchall()
        conn.close()
        for r in rows:
            self.ent_table.add_row([
                fmt_fecha(r["fecha"]), r["descripcion"],
                fmt_num(r["cantidad"]), fmt_num(r["costo_unitario"], 4),
                fmt_bs(r["costo_total"]), r["factura"]
            ])

    # ==========================================
    # TAB: SALIDAS (REQUISICIONES)
    # ==========================================
    def _build_salidas(self):
        tab = self.tabs.tab("Salidas (Requisiciones)")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)

        ctk.CTkLabel(frame, text="Registrar Salida de Material (Requisicion)",
                     font=ctk.CTkFont(size=18, weight="bold"),
                     text_color=NAVY).pack(anchor="w", pady=(0, 10))

        form = ctk.CTkFrame(frame, fg_color="transparent")
        form.pack(fill="x")

        self.sal_material = LabelCombo(form, "Material", self._get_mat_list())
        self.sal_material.pack(fill="x", pady=3)
        self.sal_cantidad = LabelEntry(form, "Cantidad", placeholder="0")
        self.sal_cantidad.pack(fill="x", pady=3)
        self.sal_fecha = LabelEntry(form, "Fecha", placeholder=fecha_hoy())
        self.sal_fecha.set(fecha_hoy())
        self.sal_fecha.pack(fill="x", pady=3)
        self.sal_nota = LabelEntry(form, "Nota/Orden", placeholder="Orden o destino")
        self.sal_nota.pack(fill="x", pady=3)

        metodo = "PEPS" if get_param("metodo_inventario") == 1 else "CPP"
        ctk.CTkLabel(form, text=f"Metodo de valuacion activo: {metodo}",
                     font=ctk.CTkFont(size=12), text_color=TEAL).pack(anchor="w", pady=5)

        ctk.CTkButton(form, text="Registrar Salida", fg_color=ORANGE,
                      width=180, command=self._save_salida).pack(pady=10, anchor="w")

        ctk.CTkLabel(frame, text="Ultimas Salidas",
                     font=ctk.CTkFont(size=14, weight="bold"),
                     text_color=BLUE).pack(anchor="w", pady=(10, 5))

        self.sal_table = ScrollableTable(
            frame,
            columns=["Fecha", "Material", "Cant.", "C.Unit.", "Total", "Nota"],
            widths=[80, 150, 60, 80, 90, 100],
            height=180
        )
        self.sal_table.pack(fill="both", expand=True)
        self._load_salidas()

    def _save_salida(self):
        mat_id = self._get_mat_id_from_combo(self.sal_material.get())
        if not mat_id:
            show_message(self, "Error", "Seleccione un material valido.", "error")
            return
        try:
            cant = float(self.sal_cantidad.get() or 0)
        except ValueError:
            show_message(self, "Error", "Cantidad debe ser numerica.", "error")
            return
        if cant <= 0:
            show_message(self, "Error", "La cantidad debe ser mayor a 0.", "error")
            return

        conn = get_connection()
        mat = conn.execute("SELECT stock_actual FROM materiales WHERE id=?", (mat_id,)).fetchone()
        if mat["stock_actual"] < cant:
            conn.close()
            show_message(self, "Error",
                         f"Stock insuficiente. Disponible: {fmt_num(mat['stock_actual'])}",
                         "error")
            return

        metodo = get_param("metodo_inventario")
        fecha = self.sal_fecha.get() or fecha_hoy()
        nota = self.sal_nota.get()

        if metodo == 1:  # PEPS
            costo_unit, costo_total = self._calcular_salida_peps(conn, mat_id, cant)
        else:  # CPP
            costo_unit, costo_total = self._calcular_salida_cpp(conn, mat_id, cant)

        cur = conn.execute("""INSERT INTO movimientos_inventario
            (material_id, tipo, cantidad, costo_unitario, costo_total, fecha, nota)
            VALUES (?,?,?,?,?,?,?)""",
            (mat_id, "salida", cant, costo_unit, costo_total, fecha, nota))
        mov_id = cur.lastrowid

        conn.execute(
            "UPDATE materiales SET stock_actual = stock_actual - ? WHERE id=?",
            (cant, mat_id))

        # Kardex
        last = conn.execute(
            "SELECT cantidad_saldo, total_saldo FROM kardex WHERE material_id=? ORDER BY id DESC LIMIT 1",
            (mat_id,)
        ).fetchone()
        saldo_cant = (last["cantidad_saldo"] if last else 0) - cant
        saldo_total = (last["total_saldo"] if last else 0) - costo_total
        saldo_unit = saldo_total / saldo_cant if saldo_cant > 0 else 0

        conn.execute("""INSERT INTO kardex
            (material_id, movimiento_id, fecha, tipo,
             cantidad_salida, costo_unit_salida, total_salida,
             cantidad_saldo, costo_unit_saldo, total_saldo)
            VALUES (?,?,?,?,?,?,?,?,?,?)""",
            (mat_id, mov_id, fecha, "salida",
             cant, costo_unit, costo_total,
             saldo_cant, saldo_unit, saldo_total))

        conn.commit()
        conn.close()

        self.sal_cantidad.set("")
        self.sal_nota.set("")
        self._load_salidas()
        self._load_materiales()
        if self.status:
            self.status.set_status(f"Salida registrada: {cant} unidades a {fmt_bs(costo_total)}")

    def _calcular_salida_peps(self, conn, mat_id, cant_requerida):
        """PEPS: Primeras Entradas, Primeras Salidas"""
        entradas = conn.execute("""
            SELECT m.id, m.cantidad, m.costo_unitario,
                   COALESCE(m.cantidad - (
                       SELECT COALESCE(SUM(k.cantidad_salida),0) FROM kardex k
                       WHERE k.material_id=m.material_id AND k.tipo='salida'
                       AND k.id > (SELECT COALESCE(MAX(k2.id),0) FROM kardex k2
                                   WHERE k2.movimiento_id=m.id)
                   ), m.cantidad) as disponible
            FROM movimientos_inventario m
            WHERE m.material_id=? AND m.tipo='entrada'
            ORDER BY m.fecha ASC, m.id ASC
        """, (mat_id,)).fetchall()

        # Fallback: usar CPP si PEPS es complejo
        return self._calcular_salida_cpp(conn, mat_id, cant_requerida)

    def _calcular_salida_cpp(self, conn, mat_id, cant):
        """CPP: Costo Promedio Ponderado"""
        last = conn.execute(
            "SELECT costo_unit_saldo FROM kardex WHERE material_id=? ORDER BY id DESC LIMIT 1",
            (mat_id,)
        ).fetchone()
        costo_unit = last["costo_unit_saldo"] if last else 0
        return costo_unit, costo_unit * cant

    def _load_salidas(self):
        self.sal_table.clear_rows()
        conn = get_connection()
        rows = conn.execute("""
            SELECT m.fecha, mat.descripcion, m.cantidad, m.costo_unitario,
                   m.costo_total, m.nota
            FROM movimientos_inventario m
            JOIN materiales mat ON m.material_id = mat.id
            WHERE m.tipo='salida'
            ORDER BY m.id DESC LIMIT 30
        """).fetchall()
        conn.close()
        for r in rows:
            self.sal_table.add_row([
                fmt_fecha(r["fecha"]), r["descripcion"],
                fmt_num(r["cantidad"]), fmt_num(r["costo_unitario"], 4),
                fmt_bs(r["costo_total"]), r["nota"] or ""
            ])

    # ==========================================
    # TAB: KARDEX
    # ==========================================
    def _build_kardex(self):
        tab = self.tabs.tab("Kardex")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)

        ctk.CTkLabel(frame, text="Kardex por Material",
                     font=ctk.CTkFont(size=18, weight="bold"),
                     text_color=NAVY).pack(anchor="w", pady=(0, 10))

        sel_frame = ctk.CTkFrame(frame, fg_color="transparent")
        sel_frame.pack(fill="x", pady=5)

        self.kardex_material = LabelCombo(sel_frame, "Material", self._get_mat_list())
        self.kardex_material.pack(side="left", padx=(0, 10))
        ctk.CTkButton(sel_frame, text="Ver Kardex", fg_color=BLUE,
                      width=120, command=self._load_kardex).pack(side="left")

        # Resumen
        self.kardex_resumen = ctk.CTkLabel(frame, text="",
                                           font=ctk.CTkFont(size=13, weight="bold"),
                                           text_color=TEAL)
        self.kardex_resumen.pack(anchor="w", pady=5)

        self.kardex_table = ScrollableTable(
            frame,
            columns=["Fecha", "Tipo", "Ent.Cant", "Ent.C.U.", "Ent.Total",
                     "Sal.Cant", "Sal.C.U.", "Sal.Total",
                     "Saldo", "C.U.Saldo", "Total Saldo"],
            widths=[70, 50, 55, 65, 70, 55, 65, 70, 55, 65, 75],
            height=350
        )
        self.kardex_table.pack(fill="both", expand=True)

    def _load_kardex(self):
        self.kardex_table.clear_rows()
        mat_id = self._get_mat_id_from_combo(self.kardex_material.get())
        if not mat_id:
            return

        conn = get_connection()
        mat = conn.execute("SELECT * FROM materiales WHERE id=?", (mat_id,)).fetchone()
        rows = conn.execute(
            "SELECT * FROM kardex WHERE material_id=? ORDER BY id", (mat_id,)
        ).fetchall()
        conn.close()

        if mat:
            self.kardex_resumen.configure(
                text=f"{mat['descripcion']} | Stock: {fmt_num(mat['stock_actual'])} {mat['unidad_medida']} | "
                     f"Precio actual: {fmt_bs(mat['precio_unitario_bs'])}")

        for r in rows:
            self.kardex_table.add_row([
                fmt_fecha(r["fecha"]),
                r["tipo"][:3].upper(),
                fmt_num(r["cantidad_entrada"]) if r["cantidad_entrada"] else "",
                fmt_num(r["costo_unit_entrada"], 4) if r["costo_unit_entrada"] else "",
                fmt_num(r["total_entrada"]) if r["total_entrada"] else "",
                fmt_num(r["cantidad_salida"]) if r["cantidad_salida"] else "",
                fmt_num(r["costo_unit_salida"], 4) if r["costo_unit_salida"] else "",
                fmt_num(r["total_salida"]) if r["total_salida"] else "",
                fmt_num(r["cantidad_saldo"]),
                fmt_num(r["costo_unit_saldo"], 4),
                fmt_num(r["total_saldo"]),
            ])
