"""
Akemi Cost Suite - M2: Mano de Obra y Nomina de Costos
Trabajadores, salario integral, cargas patronales LOTTT,
boletas de trabajo, distribucion MOD/MOI
"""
import customtkinter as ctk
from utils.database import get_connection, get_param
from utils.common import *
from datetime import datetime


class M2ManoObra(ctk.CTkFrame):
    def __init__(self, master, status_bar=None):
        super().__init__(master, fg_color="transparent")
        self.status = status_bar
        self._build_ui()

    def _build_ui(self):
        self.tabs = ctk.CTkTabview(self, anchor="nw")
        self.tabs.pack(fill="both", expand=True, padx=5, pady=5)
        self.tabs.add("Trabajadores")
        self.tabs.add("Nomina de Costos")
        self.tabs.add("Boletas de Trabajo")
        self.tabs.add("Resumen MOD/MOI")
        self._build_trabajadores()
        self._build_nomina()
        self._build_boletas()
        self._build_resumen()

    def _get_deptos(self):
        conn = get_connection()
        rows = conn.execute("SELECT nombre FROM departamentos ORDER BY nombre").fetchall()
        conn.close()
        return [r["nombre"] for r in rows] if rows else ["(sin departamentos)"]

    def _get_periodos(self):
        conn = get_connection()
        rows = conn.execute(
            "SELECT anio||'-'||printf('%02d',mes) as per FROM periodos ORDER BY anio DESC, mes DESC"
        ).fetchall()
        conn.close()
        return [r["per"] for r in rows] if rows else ["2026-02"]

    def _get_trab_list(self):
        conn = get_connection()
        rows = conn.execute(
            "SELECT cedula || ' - ' || nombre as txt FROM trabajadores WHERE activo=1 ORDER BY nombre"
        ).fetchall()
        conn.close()
        return [r["txt"] for r in rows] if rows else ["(sin trabajadores)"]

    def _get_trab_id_from_combo(self, val):
        if not val or val.startswith("("):
            return None
        cedula = val.split(" - ")[0].strip()
        conn = get_connection()
        r = conn.execute("SELECT id FROM trabajadores WHERE cedula=?", (cedula,)).fetchone()
        conn.close()
        return r["id"] if r else None

    # === TAB TRABAJADORES ===
    def _build_trabajadores(self):
        tab = self.tabs.tab("Trabajadores")
        top = ctk.CTkFrame(tab, fg_color="transparent")
        top.pack(fill="x", padx=10, pady=5)
        ctk.CTkLabel(top, text="Registro de Trabajadores",
                     font=ctk.CTkFont(size=18, weight="bold"), text_color=NAVY).pack(anchor="w", pady=(0,10))
        form = ctk.CTkFrame(top, fg_color="transparent")
        form.pack(fill="x")
        left = ctk.CTkFrame(form, fg_color="transparent")
        left.pack(side="left", fill="both", expand=True)
        right = ctk.CTkFrame(form, fg_color="transparent")
        right.pack(side="left", fill="both", expand=True, padx=(15,0))

        self.tr_cedula = LabelEntry(left, "Cedula", placeholder="V-12345678")
        self.tr_cedula.pack(fill="x", pady=3)
        self.tr_nombre = LabelEntry(left, "Nombre Completo", placeholder="Nombre y Apellido")
        self.tr_nombre.pack(fill="x", pady=3)
        self.tr_cargo = LabelEntry(left, "Cargo", placeholder="Operador, etc.")
        self.tr_cargo.pack(fill="x", pady=3)
        self.tr_depto = LabelCombo(left, "Departamento", self._get_deptos())
        self.tr_depto.pack(fill="x", pady=3)

        self.tr_tipo_mo = LabelCombo(right, "Tipo MO", ["MOD", "MOI"])
        self.tr_tipo_mo.pack(fill="x", pady=3)
        self.tr_fecha_ing = LabelEntry(right, "Fecha Ingreso", placeholder="2024-01-15")
        self.tr_fecha_ing.pack(fill="x", pady=3)
        self.tr_salario = LabelEntry(right, "Salario Basico Mensual Bs", placeholder="0.00")
        self.tr_salario.pack(fill="x", pady=3)
        self.tr_riesgo = LabelCombo(right, "Riesgo IVSS", ["minimo", "medio", "maximo"])
        self.tr_riesgo.pack(fill="x", pady=3)
        self.tr_dias_util = LabelEntry(right, "Dias Utilidades", placeholder="30")
        self.tr_dias_util.set("30")
        self.tr_dias_util.pack(fill="x", pady=3)
        self.tr_dias_bono = LabelEntry(right, "Dias Bono Vac.", placeholder="15")
        self.tr_dias_bono.set("15")
        self.tr_dias_bono.pack(fill="x", pady=3)

        btn_f = ctk.CTkFrame(top, fg_color="transparent")
        btn_f.pack(fill="x", pady=8)
        ctk.CTkButton(btn_f, text="Guardar", fg_color=GREEN, width=140,
                      command=self._save_trab).pack(side="left", padx=5)
        ctk.CTkButton(btn_f, text="Limpiar", fg_color=GRAY, width=100,
                      command=self._clear_trab).pack(side="left", padx=5)
        ctk.CTkButton(btn_f, text="Desactivar", fg_color=RED, width=100,
                      command=self._del_trab).pack(side="left", padx=5)
        self.tr_edit_id = None

        self.tr_table = ScrollableTable(tab,
            columns=["ID","Cedula","Nombre","Cargo","Tipo","Salario Bs","Depto"],
            widths=[35,85,150,100,45,90,100], height=220)
        self.tr_table.pack(fill="both", expand=True, padx=10, pady=5)
        self._load_trabs()

    def _load_trabs(self):
        self.tr_table.clear_rows()
        conn = get_connection()
        rows = conn.execute("""SELECT t.*, COALESCE(d.nombre,'') as dn
            FROM trabajadores t LEFT JOIN departamentos d ON t.departamento_id=d.id
            WHERE t.activo=1 ORDER BY t.nombre""").fetchall()
        conn.close()
        for r in rows:
            self.tr_table.add_row([r["id"], r["cedula"], r["nombre"], r["cargo"],
                r["tipo_mo"], fmt_num(r["salario_basico_mensual"]), r["dn"]],
                tag=r["id"], on_click=self._sel_trab)

    def _sel_trab(self, tid):
        conn = get_connection()
        r = conn.execute("""SELECT t.*, COALESCE(d.nombre,'') as dn
            FROM trabajadores t LEFT JOIN departamentos d ON t.departamento_id=d.id
            WHERE t.id=?""", (tid,)).fetchone()
        conn.close()
        if not r: return
        self.tr_edit_id = r["id"]
        self.tr_cedula.set(r["cedula"]); self.tr_nombre.set(r["nombre"])
        self.tr_cargo.set(r["cargo"]); self.tr_tipo_mo.set(r["tipo_mo"])
        self.tr_fecha_ing.set(r["fecha_ingreso"]); self.tr_salario.set(r["salario_basico_mensual"])
        self.tr_riesgo.set(r["riesgo_ivss"]); self.tr_dias_util.set(r["dias_utilidades"])
        self.tr_dias_bono.set(r["dias_bono_vacacional"])
        if r["dn"]: self.tr_depto.set(r["dn"])

    def _save_trab(self):
        ced = self.tr_cedula.get(); nom = self.tr_nombre.get()
        if not ced or not nom:
            show_message(self, "Error", "Cedula y Nombre obligatorios.", "error"); return
        cargo = self.tr_cargo.get(); tipo = self.tr_tipo_mo.get()
        fi = self.tr_fecha_ing.get(); riesgo = self.tr_riesgo.get()
        try: sal = float(self.tr_salario.get() or 0)
        except: sal = 0
        try: du = int(self.tr_dias_util.get() or 30)
        except: du = 30
        try: db = int(self.tr_dias_bono.get() or 15)
        except: db = 15
        conn = get_connection()
        dr = conn.execute("SELECT id FROM departamentos WHERE nombre=?", (self.tr_depto.get(),)).fetchone()
        did = dr["id"] if dr else None
        try:
            if self.tr_edit_id:
                conn.execute("""UPDATE trabajadores SET cedula=?,nombre=?,cargo=?,departamento_id=?,
                    tipo_mo=?,fecha_ingreso=?,salario_basico_mensual=?,riesgo_ivss=?,
                    dias_utilidades=?,dias_bono_vacacional=? WHERE id=?""",
                    (ced,nom,cargo,did,tipo,fi,sal,riesgo,du,db,self.tr_edit_id))
            else:
                conn.execute("""INSERT INTO trabajadores (cedula,nombre,cargo,departamento_id,
                    tipo_mo,fecha_ingreso,salario_basico_mensual,riesgo_ivss,
                    dias_utilidades,dias_bono_vacacional) VALUES (?,?,?,?,?,?,?,?,?,?)""",
                    (ced,nom,cargo,did,tipo,fi,sal,riesgo,du,db))
            conn.commit(); self._clear_trab(); self._load_trabs()
            if self.status: self.status.set_status(f"Trabajador {nom} guardado")
        except Exception as e:
            show_message(self, "Error", str(e), "error")
        finally: conn.close()

    def _del_trab(self):
        if self.tr_edit_id and confirm_dialog(self, "Confirmar", "Desactivar trabajador?"):
            conn = get_connection()
            conn.execute("UPDATE trabajadores SET activo=0 WHERE id=?", (self.tr_edit_id,))
            conn.commit(); conn.close(); self._clear_trab(); self._load_trabs()

    def _clear_trab(self):
        self.tr_edit_id = None
        for w in [self.tr_cedula, self.tr_nombre, self.tr_cargo, self.tr_fecha_ing, self.tr_salario]: w.set("")
        self.tr_dias_util.set("30"); self.tr_dias_bono.set("15")

    # === TAB NOMINA DE COSTOS ===
    def _build_nomina(self):
        tab = self.tabs.tab("Nomina de Costos")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)
        ctk.CTkLabel(frame, text="Calculo Automatico - Costo Mano de Obra",
                     font=ctk.CTkFont(size=18, weight="bold"), text_color=NAVY).pack(anchor="w", pady=(0,5))
        ctk.CTkLabel(frame, text="Salario Integral + Todas las Cargas Patronales (LOTTT/IVSS/FAOV/INCES/CEPP)",
                     font=ctk.CTkFont(size=12), text_color=TEAL).pack(anchor="w", pady=(0,10))
        sel = ctk.CTkFrame(frame, fg_color="transparent")
        sel.pack(fill="x", pady=5)
        self.nom_per = LabelCombo(sel, "Periodo", self._get_periodos())
        self.nom_per.pack(side="left", padx=(0,10))
        ctk.CTkButton(sel, text="Calcular Nomina de Costos", fg_color=GREEN,
                      width=220, command=self._calc_nomina).pack(side="left", padx=5)

        self.nom_table = ScrollableTable(frame,
            columns=["Trabajador","Tipo","Sal.Basico","Sal.Integral","Cargas","Costo Total","Costo/Hr"],
            widths=[140,40,80,85,90,95,75], height=300)
        self.nom_table.pack(fill="both", expand=True, pady=5)
        self.nom_totales = ctk.CTkLabel(frame, text="", font=ctk.CTkFont(size=14, weight="bold"), text_color=NAVY)
        self.nom_totales.pack(anchor="w", pady=5)

    def _calc_nomina(self):
        per_str = self.nom_per.get()
        try: anio, mes = int(per_str.split("-")[0]), int(per_str.split("-")[1])
        except: show_message(self, "Error", "Periodo invalido.", "error"); return

        conn = get_connection()
        per = conn.execute("SELECT id FROM periodos WHERE anio=? AND mes=?", (anio,mes)).fetchone()
        if not per:
            conn.execute("INSERT INTO periodos (anio,mes) VALUES (?,?)", (anio,mes))
            conn.commit()
            per = conn.execute("SELECT id FROM periodos WHERE anio=? AND mes=?", (anio,mes)).fetchone()
        pid = per["id"]

        params = {}
        for row in conn.execute("SELECT clave, valor FROM parametros_legales").fetchall():
            params[row["clave"]] = row["valor"]
        hrs_sem = params.get("horas_semanales", 44)
        hrs_mes = hrs_sem * 4.33

        conn.execute("DELETE FROM nomina_costos WHERE periodo_id=?", (pid,))
        trabs = conn.execute("SELECT * FROM trabajadores WHERE activo=1").fetchall()
        self.nom_table.clear_rows()
        t_mod = 0; t_moi = 0

        for t in trabs:
            sb = t["salario_basico_mensual"]
            if sb <= 0: continue
            snd = sb / 30.0
            du = t["dias_utilidades"] or int(params.get("dias_utilidades_min", 30))
            dbv = t["dias_bono_vacacional"] or int(params.get("dias_bono_vac_base", 15))
            au = (snd * du) / 360.0
            abv = (snd * dbv) / 360.0
            sid = snd + au + abv
            sim = sid * 30.0

            r = t["riesgo_ivss"] or "minimo"
            sso_p = params.get(f"sso_{r}", 9) / 100.0
            ss = sb / 4.33
            sso = ss * sso_p * 4.33
            pf = ss * (params.get("paro_forzoso", 2) / 100.0) * 4.33
            faov = sim * (params.get("faov", 2) / 100.0)
            inces = sb * (params.get("inces", 2) / 100.0)
            cepp = sb * (params.get("cepp", 9) / 100.0)
            pp = (sid * 15.0) / 3.0
            pu = (du / 360.0) * sb
            pbv = (dbv / 360.0) * sb
            ant = 0
            if t["fecha_ingreso"]:
                try:
                    fi = datetime.strptime(t["fecha_ingreso"][:10], "%Y-%m-%d")
                    ant = max(0, (datetime.now() - fi).days // 365)
                except: pass
            dv = 15 + ant
            pvc = (dv / 360.0) * sb
            tb = sso + pf + faov + inces + cepp + pp + pu + pbv + pvc
            ct = sb + tb
            ch = ct / hrs_mes if hrs_mes > 0 else 0

            conn.execute("""INSERT INTO nomina_costos
                (trabajador_id,periodo_id,salario_normal_diario,alic_utilidades,alic_bono_vac,
                 salario_integral,sso_patronal,paro_forzoso,faov,inces,cepp,
                 prov_prestaciones,prov_utilidades,prov_bono_vac,prov_vacaciones,
                 total_beneficios,costo_total_mo) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)""",
                (t["id"],pid,snd,au,abv,sid,sso,pf,faov,inces,cepp,pp,pu,pbv,pvc,tb,ct))

            self.nom_table.add_row([t["nombre"][:20], t["tipo_mo"], fmt_num(sb),
                fmt_num(sim), fmt_num(tb), fmt_num(ct), fmt_num(ch,4)],
                tag=t["id"], on_click=self._detalle_trab)
            if t["tipo_mo"] == "MOD": t_mod += ct
            else: t_moi += ct

        conn.commit(); conn.close()
        self.nom_totales.configure(text=f"MOD: {fmt_bs(t_mod)} | MOI: {fmt_bs(t_moi)} | TOTAL: {fmt_bs(t_mod+t_moi)}")
        if self.status: self.status.set_status(f"Nomina calculada: {len(trabs)} trabajadores")

    def _detalle_trab(self, tid):
        per_str = self.nom_per.get()
        try: anio, mes = int(per_str.split("-")[0]), int(per_str.split("-")[1])
        except: return
        conn = get_connection()
        n = conn.execute("""SELECT n.*, t.nombre, t.cedula, t.cargo, t.salario_basico_mensual,
            t.riesgo_ivss, t.dias_utilidades, t.dias_bono_vacacional
            FROM nomina_costos n JOIN trabajadores t ON n.trabajador_id=t.id
            WHERE n.trabajador_id=? AND n.periodo_id=(SELECT id FROM periodos WHERE anio=? AND mes=?)
        """, (tid, anio, mes)).fetchone()
        conn.close()
        if not n: return

        d = ctk.CTkToplevel(self)
        d.title(f"Detalle - {n['nombre']}"); d.geometry("550x620"); d.transient(self); d.grab_set()
        sf = ctk.CTkScrollableFrame(d, fg_color="transparent")
        sf.pack(fill="both", expand=True, padx=15, pady=15)
        ctk.CTkLabel(sf, text=f"{n['nombre']} ({n['cedula']})",
                     font=ctk.CTkFont(size=16, weight="bold"), text_color=NAVY).pack(anchor="w")
        ctk.CTkLabel(sf, text=f"Cargo: {n['cargo']} | IVSS: {n['riesgo_ivss']}",
                     font=ctk.CTkFont(size=12), text_color=GRAY).pack(anchor="w", pady=(0,10))

        def ln(lbl, val, b=False, c="333333"):
            f = ctk.CTkFrame(sf, fg_color="transparent"); f.pack(fill="x", pady=1)
            ctk.CTkLabel(f, text=lbl, width=300, anchor="w",
                font=ctk.CTkFont(size=12, weight="bold" if b else "normal"), text_color=c).pack(side="left")
            ctk.CTkLabel(f, text=val, anchor="e",
                font=ctk.CTkFont(size=12, weight="bold" if b else "normal"), text_color=c).pack(side="right")

        ln("Salario Basico Mensual", fmt_bs(n["salario_basico_mensual"]))
        ctk.CTkLabel(sf, text="--- Salario Integral (LOTTT) ---",
                     font=ctk.CTkFont(size=11, weight="bold"), text_color=BLUE).pack(anchor="w", pady=(8,2))
        ln("Salario Normal Diario", fmt_bs(n["salario_normal_diario"]))
        ln(f"Alic. Utilidades ({n['dias_utilidades']}d/360)", fmt_bs(n["alic_utilidades"]))
        ln(f"Alic. Bono Vac. ({n['dias_bono_vacacional']}d/360)", fmt_bs(n["alic_bono_vac"]))
        ln("SALARIO INTEGRAL DIARIO", fmt_bs(n["salario_integral"]), True, NAVY)
        ln("SALARIO INTEGRAL MENSUAL", fmt_bs(n["salario_integral"]*30), True, NAVY)

        ctk.CTkLabel(sf, text="--- Cargas Patronales ---",
                     font=ctk.CTkFont(size=11, weight="bold"), text_color=BLUE).pack(anchor="w", pady=(8,2))
        ln("SSO Patronal", fmt_bs(n["sso_patronal"]))
        ln("Paro Forzoso", fmt_bs(n["paro_forzoso"]))
        ln("FAOV", fmt_bs(n["faov"]))
        ln("INCES", fmt_bs(n["inces"]))
        ln("CEPP", fmt_bs(n["cepp"]))

        ctk.CTkLabel(sf, text="--- Provisiones ---",
                     font=ctk.CTkFont(size=11, weight="bold"), text_color=BLUE).pack(anchor="w", pady=(8,2))
        ln("Prov. Prestaciones (15d/trim)", fmt_bs(n["prov_prestaciones"]))
        ln("Prov. Utilidades", fmt_bs(n["prov_utilidades"]))
        ln("Prov. Bono Vacacional", fmt_bs(n["prov_bono_vac"]))
        ln("Prov. Vacaciones", fmt_bs(n["prov_vacaciones"]))

        ctk.CTkFrame(sf, height=2, fg_color=NAVY).pack(fill="x", pady=8)
        ln("TOTAL BENEFICIOS/CARGAS", fmt_bs(n["total_beneficios"]), True, ORANGE)
        ln("COSTO TOTAL MO MENSUAL", fmt_bs(n["costo_total_mo"]), True, NAVY)
        hm = get_param("horas_semanales") * 4.33
        ln("COSTO POR HORA", fmt_bs(n["costo_total_mo"]/hm if hm>0 else 0), True, TEAL)
        ctk.CTkButton(sf, text="Cerrar", fg_color=GRAY, width=100, command=d.destroy).pack(pady=15)

    # === TAB BOLETAS ===
    def _build_boletas(self):
        tab = self.tabs.tab("Boletas de Trabajo")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)
        ctk.CTkLabel(frame, text="Boletas de Trabajo (Horas por Orden/Depto)",
                     font=ctk.CTkFont(size=18, weight="bold"), text_color=NAVY).pack(anchor="w", pady=(0,10))
        form = ctk.CTkFrame(frame, fg_color="transparent"); form.pack(fill="x")
        self.bol_trab = LabelCombo(form, "Trabajador", self._get_trab_list())
        self.bol_trab.pack(fill="x", pady=3)
        self.bol_fecha = LabelEntry(form, "Fecha", placeholder=fecha_hoy())
        self.bol_fecha.set(fecha_hoy()); self.bol_fecha.pack(fill="x", pady=3)
        self.bol_horas = LabelEntry(form, "Horas", placeholder="8")
        self.bol_horas.pack(fill="x", pady=3)
        self.bol_orden = LabelEntry(form, "No. Orden (opc.)", placeholder="ORD-001")
        self.bol_orden.pack(fill="x", pady=3)
        ctk.CTkButton(form, text="Registrar Boleta", fg_color=GREEN,
                      width=160, command=self._save_bol).pack(pady=10, anchor="w")
        self.bol_table = ScrollableTable(frame,
            columns=["Fecha","Trabajador","Horas","Costo/Hr","Total","Orden"],
            widths=[80,150,55,80,80,80], height=220)
        self.bol_table.pack(fill="both", expand=True)
        self._load_bols()

    def _save_bol(self):
        tid = self._get_trab_id_from_combo(self.bol_trab.get())
        if not tid: show_message(self, "Error", "Seleccione trabajador.", "error"); return
        try: hrs = float(self.bol_horas.get() or 0)
        except: show_message(self, "Error", "Horas invalidas.", "error"); return
        if hrs <= 0: show_message(self, "Error", "Horas > 0.", "error"); return
        fecha = self.bol_fecha.get() or fecha_hoy()
        conn = get_connection()
        nm = conn.execute("SELECT costo_total_mo FROM nomina_costos WHERE trabajador_id=? ORDER BY periodo_id DESC LIMIT 1", (tid,)).fetchone()
        hm = get_param("horas_semanales") * 4.33
        if nm and hm > 0: ch = nm["costo_total_mo"] / hm
        else:
            t = conn.execute("SELECT salario_basico_mensual FROM trabajadores WHERE id=?", (tid,)).fetchone()
            ch = (t["salario_basico_mensual"] / hm) if t and hm > 0 else 0
        ct = hrs * ch
        oid = None
        ot = self.bol_orden.get()
        if ot:
            o = conn.execute("SELECT id FROM ordenes_produccion WHERE numero=?", (ot,)).fetchone()
            if o: oid = o["id"]
        conn.execute("INSERT INTO boletas_trabajo (trabajador_id,orden_id,fecha,horas,costo_hora,costo_total) VALUES (?,?,?,?,?,?)",
            (tid, oid, fecha, hrs, ch, ct))
        conn.commit(); conn.close()
        self.bol_horas.set(""); self.bol_orden.set("")
        self._load_bols()
        if self.status: self.status.set_status(f"Boleta: {hrs} hrs = {fmt_bs(ct)}")

    def _load_bols(self):
        self.bol_table.clear_rows()
        conn = get_connection()
        rows = conn.execute("""SELECT b.*, t.nombre, COALESCE(o.numero,'') as on2
            FROM boletas_trabajo b JOIN trabajadores t ON b.trabajador_id=t.id
            LEFT JOIN ordenes_produccion o ON b.orden_id=o.id
            ORDER BY b.id DESC LIMIT 50""").fetchall()
        conn.close()
        for r in rows:
            self.bol_table.add_row([fmt_fecha(r["fecha"]), r["nombre"][:20],
                fmt_num(r["horas"]), fmt_num(r["costo_hora"],4), fmt_bs(r["costo_total"]), r["on2"]])

    # === TAB RESUMEN ===
    def _build_resumen(self):
        tab = self.tabs.tab("Resumen MOD/MOI")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)
        ctk.CTkLabel(frame, text="Resumen MOD vs MOI",
                     font=ctk.CTkFont(size=18, weight="bold"), text_color=NAVY).pack(anchor="w", pady=(0,10))
        sel = ctk.CTkFrame(frame, fg_color="transparent"); sel.pack(fill="x", pady=5)
        self.res_per = LabelCombo(sel, "Periodo", self._get_periodos())
        self.res_per.pack(side="left", padx=(0,10))
        ctk.CTkButton(sel, text="Generar", fg_color=BLUE, width=120, command=self._gen_res).pack(side="left")
        self.res_frame = ctk.CTkScrollableFrame(frame, fg_color="transparent")
        self.res_frame.pack(fill="both", expand=True, pady=10)

    def _gen_res(self):
        for w in self.res_frame.winfo_children(): w.destroy()
        ps = self.res_per.get()
        try: a, m = int(ps.split("-")[0]), int(ps.split("-")[1])
        except: return
        conn = get_connection()
        rows = conn.execute("""SELECT t.tipo_mo, t.nombre, t.cargo,
            t.salario_basico_mensual, n.total_beneficios, n.costo_total_mo
            FROM nomina_costos n JOIN trabajadores t ON n.trabajador_id=t.id
            JOIN periodos p ON n.periodo_id=p.id WHERE p.anio=? AND p.mes=?
            ORDER BY t.tipo_mo, t.nombre""", (a, m)).fetchall()
        conn.close()
        if not rows:
            ctk.CTkLabel(self.res_frame, text="Sin datos. Calcule la Nomina primero.", text_color=ORANGE).pack(pady=20); return

        for tipo, color in [("MOD", GREEN), ("MOI", ORANGE)]:
            lista = [r for r in rows if r["tipo_mo"] == tipo]
            lbl = "MANO DE OBRA DIRECTA" if tipo == "MOD" else "MANO DE OBRA INDIRECTA"
            ctk.CTkLabel(self.res_frame, text=f"{lbl} ({tipo})",
                         font=ctk.CTkFont(size=14, weight="bold"), text_color=color).pack(anchor="w", pady=(10,5))
            total = 0
            for r in lista:
                f = ctk.CTkFrame(self.res_frame, fg_color=VERY_LIGHT, corner_radius=4)
                f.pack(fill="x", pady=2)
                ctk.CTkLabel(f, text=f"  {r['nombre']} - {r['cargo']}", font=ctk.CTkFont(size=12),
                             width=250, anchor="w").pack(side="left", padx=5, pady=3)
                ctk.CTkLabel(f, text=f"Basico: {fmt_bs(r['salario_basico_mensual'])}",
                             font=ctk.CTkFont(size=11), width=130).pack(side="left")
                ctk.CTkLabel(f, text=f"Cargas: {fmt_bs(r['total_beneficios'])}",
                             font=ctk.CTkFont(size=11), width=130).pack(side="left")
                ctk.CTkLabel(f, text=f"TOTAL: {fmt_bs(r['costo_total_mo'])}",
                             font=ctk.CTkFont(size=11, weight="bold"), text_color=color, width=130).pack(side="left")
                total += r["costo_total_mo"]
            if not lista:
                ctk.CTkLabel(self.res_frame, text="  (sin registros)", text_color=GRAY).pack(anchor="w")
            else:
                ctk.CTkLabel(self.res_frame, text=f"  SUBTOTAL: {fmt_bs(total)}",
                             font=ctk.CTkFont(size=13, weight="bold"), text_color=color).pack(anchor="w", pady=(5,0))

        tg = sum(r["costo_total_mo"] for r in rows)
        ctk.CTkFrame(self.res_frame, height=2, fg_color=NAVY).pack(fill="x", pady=10)
        ctk.CTkLabel(self.res_frame, text=f"TOTAL COSTO MANO DE OBRA: {fmt_bs(tg)}",
                     font=ctk.CTkFont(size=16, weight="bold"), text_color=NAVY).pack(anchor="w")
