"""
Akemi Cost Suite - M3: Costos Indirectos de Fabricacion (CIF)
Catalogo, presupuesto, tasa predeterminada, variaciones, capacidad normal vs ociosa
"""
import customtkinter as ctk
from utils.database import get_connection, get_param
from utils.common import *


class M3CIF(ctk.CTkFrame):
    def __init__(self, master, status_bar=None):
        super().__init__(master, fg_color="transparent")
        self.status = status_bar
        self._build_ui()

    def _build_ui(self):
        self.tabs = ctk.CTkTabview(self, anchor="nw")
        self.tabs.pack(fill="both", expand=True, padx=5, pady=5)
        self.tabs.add("Catalogo CIF")
        self.tabs.add("Presupuesto")
        self.tabs.add("CIF Reales")
        self.tabs.add("Variaciones")
        self._build_catalogo()
        self._build_presupuesto()
        self._build_reales()
        self._build_variaciones()

    def _get_periodos(self):
        conn = get_connection()
        rows = conn.execute("SELECT anio||'-'||printf('%02d',mes) as p FROM periodos ORDER BY anio DESC, mes DESC").fetchall()
        conn.close()
        return [r["p"] for r in rows] if rows else ["2026-02"]

    def _get_cif_list(self):
        conn = get_connection()
        rows = conn.execute("SELECT id||' - '||descripcion as t FROM cif_catalogo WHERE activo=1").fetchall()
        conn.close()
        return [r["t"] for r in rows] if rows else ["(sin CIF)"]

    def _get_deptos(self):
        conn = get_connection()
        rows = conn.execute("SELECT nombre FROM departamentos ORDER BY nombre").fetchall()
        conn.close()
        return [r["nombre"] for r in rows] if rows else ["General"]

    def _build_catalogo(self):
        tab = self.tabs.tab("Catalogo CIF")
        top = ctk.CTkFrame(tab, fg_color="transparent"); top.pack(fill="x", padx=10, pady=5)
        ctk.CTkLabel(top, text="Catalogo de Costos Indirectos de Fabricacion", font=ctk.CTkFont(size=18, weight="bold"), text_color=NAVY).pack(anchor="w", pady=(0,10))
        form = ctk.CTkFrame(top, fg_color="transparent"); form.pack(fill="x")
        self.c_desc = LabelEntry(form, "Descripcion", placeholder="Energia Electrica, Depreciacion...")
        self.c_desc.pack(fill="x", pady=3)
        self.c_tipo = LabelCombo(form, "Tipo", ["fijo","variable","semivariable"])
        self.c_tipo.pack(fill="x", pady=3)
        self.c_unidad = LabelEntry(form, "Unidad Medida", placeholder="Bs/Kwh, Bs/Lts...")
        self.c_unidad.pack(fill="x", pady=3)
        self.c_cu = LabelEntry(form, "Costo x Unidad", placeholder="0.00")
        self.c_cu.pack(fill="x", pady=3)
        self.c_cta = LabelEntry(form, "Cuenta Contable", placeholder="")
        self.c_cta.pack(fill="x", pady=3)
        bf = ctk.CTkFrame(top, fg_color="transparent"); bf.pack(fill="x", pady=8)
        ctk.CTkButton(bf, text="Guardar", fg_color=GREEN, width=120, command=self._save_cat).pack(side="left", padx=5)
        ctk.CTkButton(bf, text="Limpiar", fg_color=GRAY, width=100, command=self._clear_cat).pack(side="left", padx=5)
        self.c_eid = None
        self.c_table = ScrollableTable(tab, columns=["ID","Descripcion","Tipo","Unidad","Costo/U"], widths=[35,220,80,90,80], height=250)
        self.c_table.pack(fill="both", expand=True, padx=10, pady=5)
        self._load_cats()

    def _load_cats(self):
        self.c_table.clear_rows()
        conn = get_connection()
        for r in conn.execute("SELECT * FROM cif_catalogo WHERE activo=1 ORDER BY descripcion").fetchall():
            self.c_table.add_row([r["id"],r["descripcion"],r["tipo"],r["unidad_medida"],fmt_num(r["costo_por_unidad"],4)], tag=r["id"], on_click=self._sel_cat)
        conn.close()

    def _sel_cat(self, cid):
        conn = get_connection(); r = conn.execute("SELECT * FROM cif_catalogo WHERE id=?", (cid,)).fetchone(); conn.close()
        if not r: return
        self.c_eid = r["id"]; self.c_desc.set(r["descripcion"]); self.c_tipo.set(r["tipo"])
        self.c_unidad.set(r["unidad_medida"]); self.c_cu.set(r["costo_por_unidad"]); self.c_cta.set(r["cuenta_contable"])

    def _save_cat(self):
        d = self.c_desc.get()
        if not d: show_message(self,"Error","Descripcion obligatoria.","error"); return
        t = self.c_tipo.get(); u = self.c_unidad.get(); ct = self.c_cta.get()
        try: cu = float(self.c_cu.get() or 0)
        except: cu = 0
        conn = get_connection()
        if self.c_eid: conn.execute("UPDATE cif_catalogo SET descripcion=?,tipo=?,unidad_medida=?,costo_por_unidad=?,cuenta_contable=? WHERE id=?", (d,t,u,cu,ct,self.c_eid))
        else: conn.execute("INSERT INTO cif_catalogo (descripcion,tipo,unidad_medida,costo_por_unidad,cuenta_contable) VALUES (?,?,?,?,?)", (d,t,u,cu,ct))
        conn.commit(); conn.close(); self._clear_cat(); self._load_cats()

    def _clear_cat(self):
        self.c_eid = None
        for w in [self.c_desc,self.c_unidad,self.c_cu,self.c_cta]: w.set("")

    def _build_presupuesto(self):
        tab = self.tabs.tab("Presupuesto")
        frame = ctk.CTkFrame(tab, fg_color="transparent"); frame.pack(fill="both", expand=True, padx=10, pady=5)
        ctk.CTkLabel(frame, text="Presupuesto CIF por Periodo", font=ctk.CTkFont(size=18, weight="bold"), text_color=NAVY).pack(anchor="w", pady=(0,10))
        form = ctk.CTkFrame(frame, fg_color="transparent"); form.pack(fill="x")
        self.p_per = LabelCombo(form, "Periodo", self._get_periodos()); self.p_per.pack(fill="x", pady=3)
        self.p_cif = LabelCombo(form, "Concepto", self._get_cif_list()); self.p_cif.pack(fill="x", pady=3)
        self.p_monto = LabelEntry(form, "Monto Presup. Bs", placeholder="0.00"); self.p_monto.pack(fill="x", pady=3)
        self.p_base = LabelCombo(form, "Base Actividad", ["horas_mod","horas_maquina","costo_mod","unidades"]); self.p_base.pack(fill="x", pady=3)
        self.p_cant = LabelEntry(form, "Cant. Base Presup.", placeholder="0"); self.p_cant.pack(fill="x", pady=3)
        ctk.CTkButton(form, text="Guardar", fg_color=GREEN, width=160, command=self._save_pres).pack(pady=10, anchor="w")
        self.p_table = ScrollableTable(frame, columns=["Periodo","Concepto","Monto","Base","Cant."], widths=[80,180,100,100,80], height=200)
        self.p_table.pack(fill="both", expand=True)
        self.p_tasa = ctk.CTkLabel(frame, text="", font=ctk.CTkFont(size=14, weight="bold"), text_color=TEAL)
        self.p_tasa.pack(anchor="w", pady=5)
        self._load_pres()

    def _save_pres(self):
        ps = self.p_per.get()
        try: a,m = int(ps.split("-")[0]),int(ps.split("-")[1])
        except: return
        ct = self.p_cif.get()
        if ct.startswith("("): return
        try: cid = int(ct.split(" - ")[0])
        except: return
        try: monto = float(self.p_monto.get() or 0)
        except: monto = 0
        try: cant = float(self.p_cant.get() or 0)
        except: cant = 0
        conn = get_connection()
        per = conn.execute("SELECT id FROM periodos WHERE anio=? AND mes=?", (a,m)).fetchone()
        if not per: conn.execute("INSERT INTO periodos (anio,mes) VALUES (?,?)", (a,m)); conn.commit(); per = conn.execute("SELECT id FROM periodos WHERE anio=? AND mes=?", (a,m)).fetchone()
        conn.execute("INSERT INTO cif_presupuesto (periodo_id,cif_id,monto_presupuestado,base_actividad,cantidad_base_presup) VALUES (?,?,?,?,?)", (per["id"],cid,monto,self.p_base.get(),cant))
        conn.commit(); conn.close(); self.p_monto.set(""); self.p_cant.set(""); self._load_pres()

    def _load_pres(self):
        self.p_table.clear_rows()
        conn = get_connection()
        rows = conn.execute("""SELECT p.anio||'-'||printf('%02d',p.mes) as per, c.descripcion, cp.monto_presupuestado, cp.base_actividad, cp.cantidad_base_presup
            FROM cif_presupuesto cp JOIN periodos p ON cp.periodo_id=p.id JOIN cif_catalogo c ON cp.cif_id=c.id ORDER BY p.anio DESC, p.mes DESC""").fetchall()
        conn.close()
        tp = 0; tb = 0
        for r in rows:
            self.p_table.add_row([r["per"],r["descripcion"],fmt_bs(r["monto_presupuestado"]),r["base_actividad"],fmt_num(r["cantidad_base_presup"])])
            tp += r["monto_presupuestado"]; tb = max(tb, r["cantidad_base_presup"])
        self.p_tasa.configure(text=f"TASA PREDETERMINADA: {fmt_bs(tp/tb if tb>0 else 0)} por unidad de base" if tb > 0 else "")

    def _build_reales(self):
        tab = self.tabs.tab("CIF Reales")
        frame = ctk.CTkFrame(tab, fg_color="transparent"); frame.pack(fill="both", expand=True, padx=10, pady=5)
        ctk.CTkLabel(frame, text="CIF Reales del Periodo", font=ctk.CTkFont(size=18, weight="bold"), text_color=NAVY).pack(anchor="w", pady=(0,10))
        form = ctk.CTkFrame(frame, fg_color="transparent"); form.pack(fill="x")
        self.r_per = LabelCombo(form, "Periodo", self._get_periodos()); self.r_per.pack(fill="x", pady=3)
        self.r_cif = LabelCombo(form, "Concepto", self._get_cif_list()); self.r_cif.pack(fill="x", pady=3)
        self.r_dep = LabelCombo(form, "Departamento", self._get_deptos()); self.r_dep.pack(fill="x", pady=3)
        self.r_monto = LabelEntry(form, "Monto Real Bs", placeholder="0.00"); self.r_monto.pack(fill="x", pady=3)
        self.r_fecha = LabelEntry(form, "Fecha", placeholder=fecha_hoy()); self.r_fecha.set(fecha_hoy()); self.r_fecha.pack(fill="x", pady=3)
        self.r_desc = LabelEntry(form, "Detalle", placeholder=""); self.r_desc.pack(fill="x", pady=3)
        ctk.CTkButton(form, text="Registrar", fg_color=ORANGE, width=140, command=self._save_real).pack(pady=10, anchor="w")
        self.r_table = ScrollableTable(frame, columns=["Periodo","Concepto","Depto","Monto","Fecha"], widths=[80,180,100,100,80], height=220)
        self.r_table.pack(fill="both", expand=True); self._load_reales()

    def _save_real(self):
        ps = self.r_per.get()
        try: a,m = int(ps.split("-")[0]),int(ps.split("-")[1])
        except: return
        ct = self.r_cif.get()
        if ct.startswith("("): return
        try: cid = int(ct.split(" - ")[0])
        except: return
        try: monto = float(self.r_monto.get() or 0)
        except: monto = 0
        conn = get_connection()
        per = conn.execute("SELECT id FROM periodos WHERE anio=? AND mes=?", (a,m)).fetchone()
        if not per: return
        dn = self.r_dep.get(); dr = conn.execute("SELECT id FROM departamentos WHERE nombre=?", (dn,)).fetchone()
        did = dr["id"] if dr else None
        conn.execute("INSERT INTO cif_reales (periodo_id,cif_id,departamento_id,monto_real,fecha,descripcion) VALUES (?,?,?,?,?,?)",
            (per["id"],cid,did,monto,self.r_fecha.get(),self.r_desc.get()))
        conn.commit(); conn.close(); self.r_monto.set(""); self.r_desc.set(""); self._load_reales()

    def _load_reales(self):
        self.r_table.clear_rows()
        conn = get_connection()
        for r in conn.execute("""SELECT p.anio||'-'||printf('%02d',p.mes) as per, c.descripcion, COALESCE(d.nombre,'') as dn, cr.monto_real, cr.fecha
            FROM cif_reales cr JOIN periodos p ON cr.periodo_id=p.id JOIN cif_catalogo c ON cr.cif_id=c.id LEFT JOIN departamentos d ON cr.departamento_id=d.id
            ORDER BY cr.id DESC LIMIT 50""").fetchall():
            self.r_table.add_row([r["per"],r["descripcion"],r["dn"],fmt_bs(r["monto_real"]),fmt_fecha(r["fecha"])])
        conn.close()

    def _build_variaciones(self):
        tab = self.tabs.tab("Variaciones")
        frame = ctk.CTkFrame(tab, fg_color="transparent"); frame.pack(fill="both", expand=True, padx=10, pady=5)
        ctk.CTkLabel(frame, text="Variaciones CIF: Presupuesto vs Real", font=ctk.CTkFont(size=18, weight="bold"), text_color=NAVY).pack(anchor="w", pady=(0,10))
        sel = ctk.CTkFrame(frame, fg_color="transparent"); sel.pack(fill="x", pady=5)
        self.v_per = LabelCombo(sel, "Periodo", self._get_periodos()); self.v_per.pack(side="left", padx=(0,10))
        ctk.CTkButton(sel, text="Calcular", fg_color=BLUE, width=140, command=self._calc_var).pack(side="left")
        self.v_frame = ctk.CTkScrollableFrame(frame, fg_color="transparent"); self.v_frame.pack(fill="both", expand=True, pady=10)

    def _calc_var(self):
        for w in self.v_frame.winfo_children(): w.destroy()
        ps = self.v_per.get()
        try: a,m = int(ps.split("-")[0]),int(ps.split("-")[1])
        except: return
        conn = get_connection()
        per = conn.execute("SELECT id FROM periodos WHERE anio=? AND mes=?", (a,m)).fetchone()
        if not per: conn.close(); return
        pid = per["id"]
        tp = (conn.execute("SELECT SUM(monto_presupuestado) as s FROM cif_presupuesto WHERE periodo_id=?", (pid,)).fetchone()["s"]) or 0
        tr = (conn.execute("SELECT SUM(monto_real) as s FROM cif_reales WHERE periodo_id=?", (pid,)).fetchone()["s"]) or 0
        tf = (conn.execute("SELECT SUM(cr.monto_real) as s FROM cif_reales cr JOIN cif_catalogo c ON cr.cif_id=c.id WHERE cr.periodo_id=? AND c.tipo='fijo'", (pid,)).fetchone()["s"]) or 0
        tv = (conn.execute("SELECT SUM(cr.monto_real) as s FROM cif_reales cr JOIN cif_catalogo c ON cr.cif_id=c.id WHERE cr.periodo_id=? AND c.tipo='variable'", (pid,)).fetchone()["s"]) or 0
        conn.close()
        var = tr - tp; tv_lbl = "SOBREAPLICADO" if var < 0 else ("SUBAPLICADO" if var > 0 else "EXACTO")
        vc = GREEN if var <= 0 else RED
        def ln(l,v,b=False,c="333333"):
            f = ctk.CTkFrame(self.v_frame, fg_color="transparent"); f.pack(fill="x", pady=2)
            ctk.CTkLabel(f, text=l, width=300, anchor="w", font=ctk.CTkFont(size=13, weight="bold" if b else "normal"), text_color=c).pack(side="left")
            ctk.CTkLabel(f, text=v, anchor="e", font=ctk.CTkFont(size=13, weight="bold" if b else "normal"), text_color=c).pack(side="right")
        ln("CIF Presupuestados", fmt_bs(tp)); ln("CIF Reales", fmt_bs(tr))
        ln(f"VARIACION ({tv_lbl})", fmt_bs(abs(var)), True, vc)
        ctk.CTkFrame(self.v_frame, height=2, fg_color=BLUE).pack(fill="x", pady=10)
        ln("CIF Fijos Reales", fmt_bs(tf)); ln("CIF Variables Reales", fmt_bs(tv))
        ln("Otros/Semivariables", fmt_bs(tr-tf-tv))
