"""
Akemi Cost Suite - M4: Produccion
Departamentos, Productos/BOM, Ordenes de Produccion,
Costeo por ordenes y por procesos (departamental)
"""
import customtkinter as ctk
from utils.database import get_connection, get_param
from utils.common import *
from datetime import datetime


class M4Produccion(ctk.CTkFrame):
    def __init__(self, master, status_bar=None):
        super().__init__(master, fg_color="transparent")
        self.status = status_bar
        self._build_ui()

    def _build_ui(self):
        self.tabs = ctk.CTkTabview(self, anchor="nw")
        self.tabs.pack(fill="both", expand=True, padx=5, pady=5)
        self.tabs.add("Departamentos")
        self.tabs.add("Productos / BOM")
        self.tabs.add("Ordenes Produccion")
        self.tabs.add("Hoja de Costo")
        self._build_deptos()
        self._build_productos()
        self._build_ordenes()
        self._build_hoja_costo()

    # === HELPERS ===
    def _get_periodos(self):
        conn = get_connection()
        rows = conn.execute("SELECT anio||'-'||printf('%02d',mes) as p FROM periodos ORDER BY anio DESC, mes DESC").fetchall()
        conn.close()
        return [r["p"] for r in rows] if rows else ["2026-02"]

    # === DEPARTAMENTOS ===
    def _build_deptos(self):
        tab = self.tabs.tab("Departamentos")
        top = ctk.CTkFrame(tab, fg_color="transparent")
        top.pack(fill="x", padx=10, pady=5)
        ctk.CTkLabel(top, text="Departamentos de Produccion",
                     font=ctk.CTkFont(size=18, weight="bold"),
                     text_color=NAVY).pack(anchor="w", pady=(0, 10))
        form = ctk.CTkFrame(top, fg_color="transparent")
        form.pack(fill="x")
        self.dp_nom = LabelEntry(form, "Nombre", placeholder="Esmaltado, Impresion...")
        self.dp_nom.pack(fill="x", pady=3)
        self.dp_orden = LabelEntry(form, "Orden Secuencia", placeholder="1")
        self.dp_orden.set("1")
        self.dp_orden.pack(fill="x", pady=3)
        self.dp_cap = LabelEntry(form, "Capacidad Normal (hrs/mes)", placeholder="176")
        self.dp_cap.set("176")
        self.dp_cap.pack(fill="x", pady=3)
        self.dp_fijos = LabelCombo(form, "Recibe Costos Fijos", ["NO", "SI"])
        self.dp_fijos.pack(fill="x", pady=3)

        bf = ctk.CTkFrame(top, fg_color="transparent")
        bf.pack(fill="x", pady=8)
        ctk.CTkButton(bf, text="Guardar", fg_color=GREEN, width=120,
                      command=self._save_dep).pack(side="left", padx=5)
        ctk.CTkButton(bf, text="Limpiar", fg_color=GRAY, width=100,
                      command=self._clear_dep).pack(side="left", padx=5)
        self.dp_eid = None

        self.dp_table = ScrollableTable(tab,
            columns=["ID", "Nombre", "Orden", "Cap.Hrs", "Costos Fijos"],
            widths=[40, 200, 60, 80, 80], height=280)
        self.dp_table.pack(fill="both", expand=True, padx=10, pady=5)
        self._load_deps()

    def _load_deps(self):
        self.dp_table.clear_rows()
        conn = get_connection()
        rows = conn.execute(
            "SELECT * FROM departamentos WHERE activo=1 ORDER BY orden_secuencia"
        ).fetchall()
        conn.close()
        for r in rows:
            self.dp_table.add_row([
                r["id"], r["nombre"], r["orden_secuencia"],
                r["capacidad_normal_hrs"],
                "SI" if r["recibe_costos_fijos"] else "NO"
            ], tag=r["id"], on_click=self._sel_dep)

    def _sel_dep(self, did):
        conn = get_connection()
        r = conn.execute("SELECT * FROM departamentos WHERE id=?",
                         (did,)).fetchone()
        conn.close()
        if not r:
            return
        self.dp_eid = r["id"]
        self.dp_nom.set(r["nombre"])
        self.dp_orden.set(r["orden_secuencia"])
        self.dp_cap.set(r["capacidad_normal_hrs"])
        self.dp_fijos.set("SI" if r["recibe_costos_fijos"] else "NO")

    def _save_dep(self):
        nom = self.dp_nom.get()
        if not nom:
            show_message(self, "Error", "Nombre obligatorio.", "error")
            return
        try:
            orden = int(self.dp_orden.get() or 1)
        except:
            orden = 1
        try:
            cap = float(self.dp_cap.get() or 176)
        except:
            cap = 176
        fijos = 1 if self.dp_fijos.get() == "SI" else 0
        conn = get_connection()
        if self.dp_eid:
            conn.execute("""UPDATE departamentos SET nombre=?,
                orden_secuencia=?, capacidad_normal_hrs=?,
                recibe_costos_fijos=? WHERE id=?""",
                (nom, orden, cap, fijos, self.dp_eid))
        else:
            conn.execute("""INSERT INTO departamentos
                (nombre, orden_secuencia, capacidad_normal_hrs,
                 recibe_costos_fijos) VALUES (?,?,?,?)""",
                (nom, orden, cap, fijos))
        conn.commit()
        conn.close()
        self._clear_dep()
        self._load_deps()

    def _clear_dep(self):
        self.dp_eid = None
        self.dp_nom.set("")
        self.dp_orden.set("1")
        self.dp_cap.set("176")

    # === PRODUCTOS / BOM ===
    def _build_productos(self):
        tab = self.tabs.tab("Productos / BOM")
        top = ctk.CTkFrame(tab, fg_color="transparent")
        top.pack(fill="x", padx=10, pady=5)
        ctk.CTkLabel(top, text="Productos Terminados y Lista de Materiales",
                     font=ctk.CTkFont(size=18, weight="bold"),
                     text_color=NAVY).pack(anchor="w", pady=(0, 10))

        # Producto form
        pf = ctk.CTkFrame(top, fg_color="transparent")
        pf.pack(fill="x")
        self.pr_cod = LabelEntry(pf, "Codigo", placeholder="PROD-001")
        self.pr_cod.pack(fill="x", pady=3)
        self.pr_desc = LabelEntry(pf, "Descripcion", placeholder="Etiqueta 500ml")
        self.pr_desc.pack(fill="x", pady=3)
        self.pr_unid = LabelCombo(pf, "Unidad",
            ["Und", "Kg", "Lts", "Caja", "Resma", "Millar"])
        self.pr_unid.pack(fill="x", pady=3)

        bf = ctk.CTkFrame(top, fg_color="transparent")
        bf.pack(fill="x", pady=5)
        ctk.CTkButton(bf, text="Guardar Producto", fg_color=GREEN,
                      width=140, command=self._save_prod).pack(side="left", padx=5)
        ctk.CTkButton(bf, text="Limpiar", fg_color=GRAY, width=100,
                      command=self._clear_prod).pack(side="left", padx=5)
        self.pr_eid = None

        self.pr_table = ScrollableTable(top,
            columns=["ID", "Codigo", "Descripcion", "Unidad"],
            widths=[40, 100, 250, 80], height=120)
        self.pr_table.pack(fill="x", pady=5)
        self._load_prods()

        # BOM section
        ctk.CTkLabel(top, text="Lista de Materiales (BOM)",
                     font=ctk.CTkFont(size=14, weight="bold"),
                     text_color=BLUE).pack(anchor="w", pady=(10, 5))
        bom_f = ctk.CTkFrame(top, fg_color="transparent")
        bom_f.pack(fill="x")
        self.bom_prod = LabelCombo(bom_f, "Producto",
                                    self._get_prod_list())
        self.bom_prod.pack(fill="x", pady=3)
        self.bom_mat = LabelCombo(bom_f, "Material",
                                   self._get_mat_list())
        self.bom_mat.pack(fill="x", pady=3)
        self.bom_cant = LabelEntry(bom_f, "Cantidad/Coeficiente",
                                    placeholder="1.0")
        self.bom_cant.pack(fill="x", pady=3)
        ctk.CTkButton(bom_f, text="Agregar a BOM", fg_color=TEAL,
                      width=140, command=self._save_bom).pack(pady=5, anchor="w")

        self.bom_table = ScrollableTable(tab,
            columns=["Producto", "Material", "Cant.", "P.Unit", "Subtotal"],
            widths=[120, 180, 60, 80, 80], height=150)
        self.bom_table.pack(fill="both", expand=True, padx=10, pady=5)

    def _get_prod_list(self):
        conn = get_connection()
        rows = conn.execute(
            "SELECT codigo||' - '||descripcion as t FROM productos_terminados "
            "WHERE activo=1 ORDER BY codigo").fetchall()
        conn.close()
        return [r["t"] for r in rows] if rows else ["(sin productos)"]

    def _get_mat_list(self):
        conn = get_connection()
        rows = conn.execute(
            "SELECT codigo||' - '||descripcion as t FROM materiales "
            "WHERE activo=1 ORDER BY codigo").fetchall()
        conn.close()
        return [r["t"] for r in rows] if rows else ["(sin materiales)"]

    def _load_prods(self):
        self.pr_table.clear_rows()
        conn = get_connection()
        for r in conn.execute(
            "SELECT * FROM productos_terminados WHERE activo=1 "
            "ORDER BY codigo").fetchall():
            self.pr_table.add_row(
                [r["id"], r["codigo"], r["descripcion"],
                 r["unidad_medida"]],
                tag=r["id"], on_click=self._sel_prod)
        conn.close()

    def _sel_prod(self, pid):
        conn = get_connection()
        r = conn.execute("SELECT * FROM productos_terminados WHERE id=?",
                         (pid,)).fetchone()
        conn.close()
        if not r:
            return
        self.pr_eid = r["id"]
        self.pr_cod.set(r["codigo"])
        self.pr_desc.set(r["descripcion"])
        self.pr_unid.set(r["unidad_medida"])
        self._load_bom_for(pid)

    def _save_prod(self):
        cod = self.pr_cod.get()
        desc = self.pr_desc.get()
        if not cod or not desc:
            show_message(self, "Error", "Codigo y Descripcion requeridos.",
                         "error")
            return
        unid = self.pr_unid.get()
        conn = get_connection()
        if self.pr_eid:
            conn.execute("""UPDATE productos_terminados SET codigo=?,
                descripcion=?, unidad_medida=? WHERE id=?""",
                (cod, desc, unid, self.pr_eid))
        else:
            conn.execute("""INSERT INTO productos_terminados
                (codigo, descripcion, unidad_medida) VALUES (?,?,?)""",
                (cod, desc, unid))
        conn.commit()
        conn.close()
        self._clear_prod()
        self._load_prods()

    def _clear_prod(self):
        self.pr_eid = None
        self.pr_cod.set("")
        self.pr_desc.set("")

    def _save_bom(self):
        pt = self.bom_prod.get()
        mt = self.bom_mat.get()
        if pt.startswith("(") or mt.startswith("("):
            show_message(self, "Error",
                         "Seleccione producto y material.", "error")
            return
        try:
            cant = float(self.bom_cant.get() or 1)
        except:
            cant = 1
        conn = get_connection()
        pc = pt.split(" - ")[0].strip()
        mc = mt.split(" - ")[0].strip()
        pr = conn.execute(
            "SELECT id FROM productos_terminados WHERE codigo=?",
            (pc,)).fetchone()
        mr = conn.execute(
            "SELECT id FROM materiales WHERE codigo=?",
            (mc,)).fetchone()
        if pr and mr:
            conn.execute("""INSERT INTO bom
                (producto_id, material_id, cantidad_estandar)
                VALUES (?,?,?)""", (pr["id"], mr["id"], cant))
            conn.commit()
            self._load_bom_for(pr["id"])
        conn.close()
        self.bom_cant.set("")

    def _load_bom_for(self, prod_id):
        self.bom_table.clear_rows()
        conn = get_connection()
        rows = conn.execute("""
            SELECT pt.codigo as pc, m.descripcion as md,
                   b.cantidad_estandar, m.precio_unitario_bs
            FROM bom b
            JOIN productos_terminados pt ON b.producto_id=pt.id
            JOIN materiales m ON b.material_id=m.id
            WHERE b.producto_id=? ORDER BY m.descripcion
        """, (prod_id,)).fetchall()
        conn.close()
        for r in rows:
            sub = r["cantidad_estandar"] * r["precio_unitario_bs"]
            self.bom_table.add_row([
                r["pc"], r["md"], fmt_num(r["cantidad_estandar"], 4),
                fmt_num(r["precio_unitario_bs"]), fmt_bs(sub)])

    # === ORDENES DE PRODUCCION ===
    def _build_ordenes(self):
        tab = self.tabs.tab("Ordenes Produccion")
        top = ctk.CTkFrame(tab, fg_color="transparent")
        top.pack(fill="x", padx=10, pady=5)
        ctk.CTkLabel(top, text="Ordenes de Produccion",
                     font=ctk.CTkFont(size=18, weight="bold"),
                     text_color=NAVY).pack(anchor="w", pady=(0, 10))
        form = ctk.CTkFrame(top, fg_color="transparent")
        form.pack(fill="x")
        self.or_num = LabelEntry(form, "Numero Orden",
                                  placeholder="ORD-001")
        self.or_num.pack(fill="x", pady=3)
        self.or_prod = LabelCombo(form, "Producto",
                                   self._get_prod_list())
        self.or_prod.pack(fill="x", pady=3)
        self.or_cant = LabelEntry(form, "Cantidad a Producir",
                                   placeholder="100")
        self.or_cant.pack(fill="x", pady=3)
        self.or_fi = LabelEntry(form, "Fecha Inicio",
                                 placeholder=fecha_hoy())
        self.or_fi.set(fecha_hoy())
        self.or_fi.pack(fill="x", pady=3)
        self.or_per = LabelCombo(form, "Periodo",
                                  self._get_periodos())
        self.or_per.pack(fill="x", pady=3)
        self.or_estado = LabelCombo(form, "Estado",
            ["abierta", "en_proceso", "terminada", "cerrada"])
        self.or_estado.pack(fill="x", pady=3)

        bf = ctk.CTkFrame(top, fg_color="transparent")
        bf.pack(fill="x", pady=8)
        ctk.CTkButton(bf, text="Guardar Orden", fg_color=GREEN,
                      width=140, command=self._save_ord).pack(side="left", padx=5)
        ctk.CTkButton(bf, text="Limpiar", fg_color=GRAY, width=100,
                      command=self._clear_ord).pack(side="left", padx=5)
        self.or_eid = None

        self.or_table = ScrollableTable(tab,
            columns=["ID", "Numero", "Producto", "Cant.", "Estado",
                     "Fecha"],
            widths=[35, 80, 180, 60, 80, 80], height=260)
        self.or_table.pack(fill="both", expand=True, padx=10, pady=5)
        self._load_ords()

    def _load_ords(self):
        self.or_table.clear_rows()
        conn = get_connection()
        rows = conn.execute("""
            SELECT o.*, pt.descripcion as pd
            FROM ordenes_produccion o
            JOIN productos_terminados pt ON o.producto_id=pt.id
            ORDER BY o.id DESC LIMIT 50
        """).fetchall()
        conn.close()
        for r in rows:
            self.or_table.add_row([
                r["id"], r["numero"], r["pd"][:25],
                fmt_num(r["cantidad"]), r["estado"],
                fmt_fecha(r["fecha_inicio"])
            ], tag=r["id"], on_click=self._sel_ord)

    def _sel_ord(self, oid):
        conn = get_connection()
        r = conn.execute("""SELECT o.*, pt.codigo||' - '||pt.descripcion as pt
            FROM ordenes_produccion o
            JOIN productos_terminados pt ON o.producto_id=pt.id
            WHERE o.id=?""", (oid,)).fetchone()
        conn.close()
        if not r:
            return
        self.or_eid = r["id"]
        self.or_num.set(r["numero"])
        self.or_prod.set(r["pt"])
        self.or_cant.set(r["cantidad"])
        self.or_fi.set(r["fecha_inicio"])
        self.or_estado.set(r["estado"])

    def _save_ord(self):
        num = self.or_num.get()
        if not num:
            show_message(self, "Error", "Numero de orden requerido.",
                         "error")
            return
        pt = self.or_prod.get()
        if pt.startswith("("):
            show_message(self, "Error", "Seleccione producto.", "error")
            return
        pc = pt.split(" - ")[0].strip()
        try:
            cant = float(self.or_cant.get() or 1)
        except:
            cant = 1
        fi = self.or_fi.get() or fecha_hoy()
        estado = self.or_estado.get()
        ps = self.or_per.get()
        try:
            a, m = int(ps.split("-")[0]), int(ps.split("-")[1])
        except:
            a, m = 2026, 2

        conn = get_connection()
        pr = conn.execute(
            "SELECT id FROM productos_terminados WHERE codigo=?",
            (pc,)).fetchone()
        if not pr:
            show_message(self, "Error", "Producto no encontrado.",
                         "error")
            conn.close()
            return
        per = conn.execute(
            "SELECT id FROM periodos WHERE anio=? AND mes=?",
            (a, m)).fetchone()
        if not per:
            conn.execute("INSERT INTO periodos (anio,mes) VALUES (?,?)",
                         (a, m))
            conn.commit()
            per = conn.execute(
                "SELECT id FROM periodos WHERE anio=? AND mes=?",
                (a, m)).fetchone()
        pid = per["id"]

        if self.or_eid:
            conn.execute("""UPDATE ordenes_produccion SET numero=?,
                producto_id=?, cantidad=?, fecha_inicio=?,
                periodo_id=?, estado=? WHERE id=?""",
                (num, pr["id"], cant, fi, pid, estado, self.or_eid))
        else:
            conn.execute("""INSERT INTO ordenes_produccion
                (numero, producto_id, cantidad, fecha_inicio,
                 periodo_id, estado) VALUES (?,?,?,?,?,?)""",
                (num, pr["id"], cant, fi, pid, estado))
        conn.commit()
        conn.close()
        self._clear_ord()
        self._load_ords()

    def _clear_ord(self):
        self.or_eid = None
        self.or_num.set("")
        self.or_cant.set("")

    # === HOJA DE COSTO ===
    def _build_hoja_costo(self):
        tab = self.tabs.tab("Hoja de Costo")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)
        ctk.CTkLabel(frame, text="Hoja de Costo por Orden",
                     font=ctk.CTkFont(size=18, weight="bold"),
                     text_color=NAVY).pack(anchor="w", pady=(0, 5))
        ctk.CTkLabel(frame,
            text="Estructura: MD + MOD + CIF = Costo Produccion",
            font=ctk.CTkFont(size=12),
            text_color=TEAL).pack(anchor="w", pady=(0, 10))

        sel = ctk.CTkFrame(frame, fg_color="transparent")
        sel.pack(fill="x", pady=5)
        self.hc_ord = LabelCombo(sel, "Orden",
                                  self._get_ord_list())
        self.hc_ord.pack(side="left", padx=(0, 10))
        ctk.CTkButton(sel, text="Calcular Hoja de Costo",
                      fg_color=BLUE, width=200,
                      command=self._calc_hoja).pack(side="left")

        self.hc_frame = ctk.CTkScrollableFrame(frame,
                                                fg_color="transparent")
        self.hc_frame.pack(fill="both", expand=True, pady=10)

    def _get_ord_list(self):
        conn = get_connection()
        rows = conn.execute("""
            SELECT o.numero||' - '||pt.descripcion as t
            FROM ordenes_produccion o
            JOIN productos_terminados pt ON o.producto_id=pt.id
            ORDER BY o.id DESC""").fetchall()
        conn.close()
        return [r["t"] for r in rows] if rows else ["(sin ordenes)"]

    def _calc_hoja(self):
        for w in self.hc_frame.winfo_children():
            w.destroy()
        ot = self.hc_ord.get()
        if ot.startswith("("):
            return
        num = ot.split(" - ")[0].strip()
        conn = get_connection()
        orden = conn.execute("""
            SELECT o.*, pt.descripcion as pd, pt.codigo as pc
            FROM ordenes_produccion o
            JOIN productos_terminados pt ON o.producto_id=pt.id
            WHERE o.numero=?""", (num,)).fetchone()
        if not orden:
            conn.close()
            return

        oid = orden["id"]
        pid = orden["producto_id"]
        cant = orden["cantidad"]

        def section(title, color=NAVY):
            ctk.CTkLabel(self.hc_frame, text=title,
                         font=ctk.CTkFont(size=14, weight="bold"),
                         text_color=color).pack(anchor="w", pady=(10, 5))

        def ln(lbl, val, b=False, c="333333"):
            f = ctk.CTkFrame(self.hc_frame, fg_color="transparent")
            f.pack(fill="x", pady=1)
            ctk.CTkLabel(f, text=lbl, width=350, anchor="w",
                font=ctk.CTkFont(size=12,
                    weight="bold" if b else "normal"),
                text_color=c).pack(side="left")
            ctk.CTkLabel(f, text=val, anchor="e",
                font=ctk.CTkFont(size=12,
                    weight="bold" if b else "normal"),
                text_color=c).pack(side="right")

        # Header
        ctk.CTkLabel(self.hc_frame,
            text=f"HOJA DE COSTO - Orden {orden['numero']}",
            font=ctk.CTkFont(size=16, weight="bold"),
            text_color=NAVY).pack(anchor="w")
        ln("Producto", f"{orden['pc']} - {orden['pd']}")
        ln("Cantidad", fmt_num(cant))
        ln("Estado", orden["estado"])

        # 1. MATERIALES DIRECTOS
        section("1. MATERIALES DIRECTOS", GREEN)
        # From salidas de inventario linked to this order
        mats = conn.execute("""
            SELECT m.descripcion, mi.cantidad, mi.costo_unitario,
                   mi.costo_total
            FROM movimientos_inventario mi
            JOIN materiales m ON mi.material_id=m.id
            WHERE mi.tipo='salida' AND mi.nota_orden=?
            ORDER BY mi.fecha
        """, (num,)).fetchall()

        total_md = 0
        if mats:
            for r in mats:
                ln(f"  {r['descripcion']}  x{fmt_num(r['cantidad'])}",
                   fmt_bs(r["costo_total"]))
                total_md += r["costo_total"]
        else:
            # Estimate from BOM
            bom = conn.execute("""
                SELECT m.descripcion, b.cantidad_estandar,
                       m.precio_unitario_bs
                FROM bom b JOIN materiales m ON b.material_id=m.id
                WHERE b.producto_id=?""", (pid,)).fetchall()
            for r in bom:
                sub = r["cantidad_estandar"] * cant * r["precio_unitario_bs"]
                ln(f"  {r['descripcion']}  x{fmt_num(r['cantidad_estandar']*cant)}",
                   fmt_bs(sub))
                total_md += sub
            if bom:
                ctk.CTkLabel(self.hc_frame,
                    text="  (estimado desde BOM - sin salidas registradas)",
                    font=ctk.CTkFont(size=10),
                    text_color=ORANGE).pack(anchor="w")
        ln("TOTAL MATERIALES DIRECTOS", fmt_bs(total_md), True, GREEN)

        # 2. MANO DE OBRA DIRECTA
        section("2. MANO DE OBRA DIRECTA", BLUE)
        bols = conn.execute("""
            SELECT t.nombre, b.horas, b.costo_hora, b.costo_total
            FROM boletas_trabajo b
            JOIN trabajadores t ON b.trabajador_id=t.id
            WHERE b.orden_id=? ORDER BY b.fecha
        """, (oid,)).fetchall()
        total_mod = 0
        for r in bols:
            ln(f"  {r['nombre']}  {fmt_num(r['horas'])} hrs",
               fmt_bs(r["costo_total"]))
            total_mod += r["costo_total"]
        if not bols:
            ctk.CTkLabel(self.hc_frame,
                text="  (sin boletas de trabajo asignadas)",
                font=ctk.CTkFont(size=10),
                text_color=ORANGE).pack(anchor="w")
        ln("TOTAL MOD", fmt_bs(total_mod), True, BLUE)

        # 3. CIF APLICADOS
        section("3. CIF APLICADOS", ORANGE)
        # Use predetermined rate from budget
        per_id = orden["periodo_id"]
        pres = conn.execute("""
            SELECT SUM(monto_presupuestado) as tp,
                   MAX(cantidad_base_presup) as tb,
                   MAX(base_actividad) as ba
            FROM cif_presupuesto WHERE periodo_id=?
        """, (per_id,)).fetchone()

        total_cif = 0
        if pres and pres["tp"] and pres["tb"] and pres["tb"] > 0:
            tasa = pres["tp"] / pres["tb"]
            base_name = pres["ba"] or "unidades"
            if base_name == "horas_mod":
                total_hrs = sum(r["horas"] for r in bols) if bols else 0
                total_cif = tasa * total_hrs
                ln(f"  Tasa: {fmt_bs(tasa)}/hr x {fmt_num(total_hrs)} hrs",
                   fmt_bs(total_cif))
            else:
                total_cif = tasa * cant
                ln(f"  Tasa: {fmt_bs(tasa)}/ud x {fmt_num(cant)} uds",
                   fmt_bs(total_cif))
        else:
            # Prorrateo simple: CIF reales / ordenes
            cif_r = conn.execute("""
                SELECT SUM(monto_real) as t FROM cif_reales
                WHERE periodo_id=?""", (per_id,)).fetchone()
            n_ord = conn.execute("""
                SELECT COUNT(*) as c FROM ordenes_produccion
                WHERE periodo_id=?""", (per_id,)).fetchone()
            if cif_r["t"] and n_ord["c"] > 0:
                total_cif = cif_r["t"] / n_ord["c"]
                ln("  (Prorrateo simple CIF reales/ordenes)",
                   fmt_bs(total_cif))
        ln("TOTAL CIF APLICADOS", fmt_bs(total_cif), True, ORANGE)

        conn.close()

        # RESUMEN
        ctk.CTkFrame(self.hc_frame, height=3,
                     fg_color=NAVY).pack(fill="x", pady=10)
        costo_prod = total_md + total_mod + total_cif
        costo_unit = costo_prod / cant if cant > 0 else 0
        ln("COSTO DE PRODUCCION TOTAL", fmt_bs(costo_prod), True, NAVY)
        ln("COSTO UNITARIO", fmt_bs(costo_unit), True, TEAL)

        if self.status:
            self.status.set_status(
                f"Hoja de costo {num}: {fmt_bs(costo_prod)}")
