"""
Akemi Cost Suite - M5: Estado de Costos y Reportes
Estado de Costo de Produccion y Ventas,
Dashboard de costos, reportes por periodo
"""
import customtkinter as ctk
from utils.database import get_connection, get_param
from utils.common import *


class M5Reportes(ctk.CTkFrame):
    def __init__(self, master, status_bar=None):
        super().__init__(master, fg_color="transparent")
        self.status = status_bar
        self._build_ui()

    def _build_ui(self):
        self.tabs = ctk.CTkTabview(self, anchor="nw")
        self.tabs.pack(fill="both", expand=True, padx=5, pady=5)
        self.tabs.add("Estado de Costos")
        self.tabs.add("Dashboard")
        self.tabs.add("Comparativo")
        self._build_estado()
        self._build_dashboard()
        self._build_comparativo()

    def _get_periodos(self):
        conn = get_connection()
        rows = conn.execute(
            "SELECT anio||'-'||printf('%02d',mes) as p "
            "FROM periodos ORDER BY anio DESC, mes DESC"
        ).fetchall()
        conn.close()
        return [r["p"] for r in rows] if rows else ["2026-02"]

    # === ESTADO DE COSTOS ===
    def _build_estado(self):
        tab = self.tabs.tab("Estado de Costos")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)
        ctk.CTkLabel(frame,
            text="Estado de Costo de Produccion y Ventas",
            font=ctk.CTkFont(size=18, weight="bold"),
            text_color=NAVY).pack(anchor="w", pady=(0, 5))
        ctk.CTkLabel(frame,
            text="Formato NIC 2 / Seccion 13 NIIF PYMES",
            font=ctk.CTkFont(size=12),
            text_color=TEAL).pack(anchor="w", pady=(0, 10))

        sel = ctk.CTkFrame(frame, fg_color="transparent")
        sel.pack(fill="x", pady=5)
        self.ec_per = LabelCombo(sel, "Periodo",
                                  self._get_periodos())
        self.ec_per.pack(side="left", padx=(0, 10))
        ctk.CTkButton(sel, text="Generar Estado de Costos",
                      fg_color=GREEN, width=220,
                      command=self._gen_estado).pack(side="left")

        self.ec_frame = ctk.CTkScrollableFrame(frame,
                                                fg_color="transparent")
        self.ec_frame.pack(fill="both", expand=True, pady=10)

    def _gen_estado(self):
        for w in self.ec_frame.winfo_children():
            w.destroy()
        ps = self.ec_per.get()
        try:
            a, m = int(ps.split("-")[0]), int(ps.split("-")[1])
        except:
            return
        conn = get_connection()
        per = conn.execute(
            "SELECT id FROM periodos WHERE anio=? AND mes=?",
            (a, m)).fetchone()
        if not per:
            ctk.CTkLabel(self.ec_frame,
                text="Periodo no encontrado.",
                text_color=RED).pack(pady=20)
            conn.close()
            return
        pid = per["id"]

        emp = conn.execute(
            "SELECT razon_social FROM empresa LIMIT 1").fetchone()
        empresa = emp["razon_social"] if emp else "Empresa"

        def title(t, c=NAVY):
            ctk.CTkLabel(self.ec_frame, text=t,
                font=ctk.CTkFont(size=14, weight="bold"),
                text_color=c).pack(anchor="w", pady=(8, 3))

        def ln(lbl, val, b=False, c="333333", indent=0):
            f = ctk.CTkFrame(self.ec_frame, fg_color="transparent")
            f.pack(fill="x", pady=1)
            pref = "  " * indent
            ctk.CTkLabel(f, text=pref + lbl, width=380, anchor="w",
                font=ctk.CTkFont(size=12,
                    weight="bold" if b else "normal"),
                text_color=c).pack(side="left")
            ctk.CTkLabel(f, text=val, anchor="e",
                font=ctk.CTkFont(size=12,
                    weight="bold" if b else "normal"),
                text_color=c).pack(side="right")

        def sep():
            ctk.CTkFrame(self.ec_frame, height=1,
                         fg_color=GRAY).pack(fill="x", pady=4)

        # Header
        ctk.CTkLabel(self.ec_frame, text=empresa,
            font=ctk.CTkFont(size=16, weight="bold"),
            text_color=NAVY).pack(anchor="w")
        ctk.CTkLabel(self.ec_frame,
            text=f"ESTADO DE COSTO DE PRODUCCION Y VENTAS",
            font=ctk.CTkFont(size=14, weight="bold"),
            text_color=NAVY).pack(anchor="w")
        ctk.CTkLabel(self.ec_frame,
            text=f"Periodo: {ps}",
            font=ctk.CTkFont(size=12),
            text_color=GRAY).pack(anchor="w", pady=(0, 10))

        # 1. MATERIALES DIRECTOS CONSUMIDOS
        title("1. MATERIALES DIRECTOS CONSUMIDOS")

        # Inv inicial (simplificado: stock al inicio)
        inv_ini_md = 0  # Would need prior period data
        compras = conn.execute("""
            SELECT COALESCE(SUM(mi.costo_total),0) as t
            FROM movimientos_inventario mi
            JOIN materiales m ON mi.material_id=m.id
            WHERE mi.tipo='entrada' AND mi.fecha LIKE ?
            AND m.tipo='directo'
        """, (f"{a}-{m:02d}%",)).fetchone()["t"]

        consumo_md = conn.execute("""
            SELECT COALESCE(SUM(mi.costo_total),0) as t
            FROM movimientos_inventario mi
            JOIN materiales m ON mi.material_id=m.id
            WHERE mi.tipo='salida' AND mi.fecha LIKE ?
            AND m.tipo='directo'
        """, (f"{a}-{m:02d}%",)).fetchone()["t"]

        ln("Inventario Inicial MD", fmt_bs(inv_ini_md), False, "333", 1)
        ln("(+) Compras del periodo", fmt_bs(compras), False, "333", 1)
        ln("(-) Inventario Final MD", fmt_bs(0), False, "333", 1)
        ln("(=) MD Consumidos", fmt_bs(consumo_md), True, GREEN)

        sep()

        # 2. MANO DE OBRA DIRECTA
        title("2. MANO DE OBRA DIRECTA")
        mod_total = conn.execute("""
            SELECT COALESCE(SUM(n.costo_total_mo),0) as t
            FROM nomina_costos n
            JOIN trabajadores t ON n.trabajador_id=t.id
            WHERE n.periodo_id=? AND t.tipo_mo='MOD'
        """, (pid,)).fetchone()["t"]
        ln("Total MOD (Nomina de Costos)", fmt_bs(mod_total),
           True, BLUE)

        sep()

        # 3. CIF
        title("3. COSTOS INDIRECTOS DE FABRICACION")
        # MOI
        moi_total = conn.execute("""
            SELECT COALESCE(SUM(n.costo_total_mo),0) as t
            FROM nomina_costos n
            JOIN trabajadores t ON n.trabajador_id=t.id
            WHERE n.periodo_id=? AND t.tipo_mo='MOI'
        """, (pid,)).fetchone()["t"]
        # Materiales indirectos
        mi_total = conn.execute("""
            SELECT COALESCE(SUM(mi.costo_total),0) as t
            FROM movimientos_inventario mi
            JOIN materiales m ON mi.material_id=m.id
            WHERE mi.tipo='salida' AND mi.fecha LIKE ?
            AND m.tipo='indirecto'
        """, (f"{a}-{m:02d}%",)).fetchone()["t"]
        # CIF reales
        cif_otros = conn.execute("""
            SELECT COALESCE(SUM(monto_real),0) as t
            FROM cif_reales WHERE periodo_id=?
        """, (pid,)).fetchone()["t"]

        cif_total = moi_total + mi_total + cif_otros
        ln("MOI (Nomina)", fmt_bs(moi_total), False, "333", 1)
        ln("Materiales Indirectos", fmt_bs(mi_total), False, "333", 1)
        ln("Otros CIF", fmt_bs(cif_otros), False, "333", 1)
        ln("Total CIF", fmt_bs(cif_total), True, ORANGE)

        sep()

        # COSTO DE PRODUCCION
        costo_prod = consumo_md + mod_total + cif_total
        ctk.CTkFrame(self.ec_frame, height=3,
                     fg_color=NAVY).pack(fill="x", pady=8)
        ln("COSTO DE PRODUCCION DEL PERIODO",
           fmt_bs(costo_prod), True, NAVY)

        # Inv PT
        title("Inventario Productos Terminados")
        ln("(+) Inv. Inicial PT", fmt_bs(0), False, "333", 1)
        ln("(-) Inv. Final PT", fmt_bs(0), False, "333", 1)

        ctk.CTkFrame(self.ec_frame, height=3,
                     fg_color=NAVY).pack(fill="x", pady=8)
        ln("COSTO DE PRODUCCION Y VENTAS",
           fmt_bs(costo_prod), True, NAVY)

        conn.close()
        if self.status:
            self.status.set_status(
                f"Estado de Costos generado: {fmt_bs(costo_prod)}")

    # === DASHBOARD ===
    def _build_dashboard(self):
        tab = self.tabs.tab("Dashboard")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)
        ctk.CTkLabel(frame, text="Dashboard de Costos",
                     font=ctk.CTkFont(size=18, weight="bold"),
                     text_color=NAVY).pack(anchor="w", pady=(0, 10))
        sel = ctk.CTkFrame(frame, fg_color="transparent")
        sel.pack(fill="x", pady=5)
        self.db_per = LabelCombo(sel, "Periodo",
                                  self._get_periodos())
        self.db_per.pack(side="left", padx=(0, 10))
        ctk.CTkButton(sel, text="Actualizar",
                      fg_color=BLUE, width=120,
                      command=self._gen_dash).pack(side="left")
        self.db_frame = ctk.CTkFrame(frame, fg_color="transparent")
        self.db_frame.pack(fill="both", expand=True, pady=10)

    def _gen_dash(self):
        for w in self.db_frame.winfo_children():
            w.destroy()
        ps = self.db_per.get()
        try:
            a, m = int(ps.split("-")[0]), int(ps.split("-")[1])
        except:
            return
        conn = get_connection()
        per = conn.execute(
            "SELECT id FROM periodos WHERE anio=? AND mes=?",
            (a, m)).fetchone()
        if not per:
            conn.close()
            return
        pid = per["id"]

        # Cards
        cards = ctk.CTkFrame(self.db_frame, fg_color="transparent")
        cards.pack(fill="x", pady=10)

        def card(parent, title, value, color):
            c = ctk.CTkFrame(parent, fg_color=color,
                             corner_radius=8, width=200, height=80)
            c.pack(side="left", padx=8, pady=5)
            c.pack_propagate(False)
            ctk.CTkLabel(c, text=title,
                font=ctk.CTkFont(size=11),
                text_color="white").pack(pady=(10, 0))
            ctk.CTkLabel(c, text=value,
                font=ctk.CTkFont(size=16, weight="bold"),
                text_color="white").pack()

        # MD
        md = conn.execute("""
            SELECT COALESCE(SUM(costo_total),0) as t
            FROM movimientos_inventario
            WHERE tipo='salida' AND fecha LIKE ?
        """, (f"{a}-{m:02d}%",)).fetchone()["t"]

        # MOD
        mod = conn.execute("""
            SELECT COALESCE(SUM(n.costo_total_mo),0) as t
            FROM nomina_costos n JOIN trabajadores t
            ON n.trabajador_id=t.id
            WHERE n.periodo_id=? AND t.tipo_mo='MOD'
        """, (pid,)).fetchone()["t"]

        # CIF
        cif = conn.execute("""
            SELECT COALESCE(SUM(monto_real),0) as t
            FROM cif_reales WHERE periodo_id=?
        """, (pid,)).fetchone()["t"]

        # Ordenes
        ords = conn.execute("""
            SELECT COUNT(*) as c FROM ordenes_produccion
            WHERE periodo_id=?""", (pid,)).fetchone()["c"]

        total = md + mod + cif
        card(cards, "Materiales", fmt_bs(md), GREEN)
        card(cards, "Mano Obra", fmt_bs(mod), BLUE)
        card(cards, "CIF", fmt_bs(cif), ORANGE)
        card(cards, "Total", fmt_bs(total), NAVY)

        # Composition
        comp = ctk.CTkFrame(self.db_frame, fg_color="transparent")
        comp.pack(fill="x", pady=10)
        ctk.CTkLabel(comp, text="Composicion del Costo",
                     font=ctk.CTkFont(size=14, weight="bold"),
                     text_color=NAVY).pack(anchor="w", pady=5)

        if total > 0:
            pcts = [
                ("Materiales Directos",
                 md / total * 100, GREEN),
                ("Mano de Obra Directa",
                 mod / total * 100, BLUE),
                ("CIF", cif / total * 100, ORANGE)
            ]
            for name, pct, color in pcts:
                rf = ctk.CTkFrame(comp, fg_color="transparent")
                rf.pack(fill="x", pady=2)
                ctk.CTkLabel(rf, text=f"{name}: {pct:.1f}%",
                    width=200, anchor="w",
                    font=ctk.CTkFont(size=12)).pack(side="left")
                bar = ctk.CTkProgressBar(rf, width=300,
                    progress_color=color)
                bar.pack(side="left", padx=10)
                bar.set(pct / 100)
                ctk.CTkLabel(rf, text=fmt_bs(
                    md if name.startswith("Mat") else (
                    mod if name.startswith("Mano") else cif)),
                    font=ctk.CTkFont(size=12),
                    text_color=color).pack(side="left")

        # Info
        ctk.CTkLabel(self.db_frame,
            text=f"Ordenes del periodo: {ords}",
            font=ctk.CTkFont(size=12),
            text_color=GRAY).pack(anchor="w", pady=10)

        conn.close()

    # === COMPARATIVO ===
    def _build_comparativo(self):
        tab = self.tabs.tab("Comparativo")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)
        ctk.CTkLabel(frame, text="Comparativo Multi-Periodo",
                     font=ctk.CTkFont(size=18, weight="bold"),
                     text_color=NAVY).pack(anchor="w", pady=(0, 10))
        ctk.CTkButton(frame, text="Generar Comparativo (ultimos 6 periodos)",
                      fg_color=BLUE, width=280,
                      command=self._gen_comp).pack(anchor="w", pady=10)
        self.comp_table = ScrollableTable(frame,
            columns=["Periodo", "MD", "MOD", "CIF", "Total",
                     "Ordenes"],
            widths=[80, 100, 100, 100, 100, 60], height=300)
        self.comp_table.pack(fill="both", expand=True)

    def _gen_comp(self):
        self.comp_table.clear_rows()
        conn = get_connection()
        periodos = conn.execute("""
            SELECT id, anio, mes,
                   anio||'-'||printf('%02d',mes) as per
            FROM periodos ORDER BY anio DESC, mes DESC LIMIT 6
        """).fetchall()

        for p in periodos:
            pid = p["id"]
            per = p["per"]
            prefix = f"{p['anio']}-{p['mes']:02d}%"

            md = conn.execute("""
                SELECT COALESCE(SUM(costo_total),0) as t
                FROM movimientos_inventario
                WHERE tipo='salida' AND fecha LIKE ?
            """, (prefix,)).fetchone()["t"]

            mod = conn.execute("""
                SELECT COALESCE(SUM(n.costo_total_mo),0) as t
                FROM nomina_costos n JOIN trabajadores t
                ON n.trabajador_id=t.id
                WHERE n.periodo_id=? AND t.tipo_mo='MOD'
            """, (pid,)).fetchone()["t"]

            cif = conn.execute("""
                SELECT COALESCE(SUM(monto_real),0) as t
                FROM cif_reales WHERE periodo_id=?
            """, (pid,)).fetchone()["t"]

            ords = conn.execute("""
                SELECT COUNT(*) as c FROM ordenes_produccion
                WHERE periodo_id=?""", (pid,)).fetchone()["c"]

            total = md + mod + cif
            self.comp_table.add_row([
                per, fmt_bs(md), fmt_bs(mod), fmt_bs(cif),
                fmt_bs(total), str(ords)])

        conn.close()
