"""
Akemi Cost Suite - M6: Estructura de Costos SUNDDE
LOPJ Art. 13-15, margen maximo 30%, formatos SUNDDE,
estructura de costos para precios regulados
"""
import customtkinter as ctk
from utils.database import get_connection, get_param
from utils.common import *


class M6SUNDDE(ctk.CTkFrame):
    def __init__(self, master, status_bar=None):
        super().__init__(master, fg_color="transparent")
        self.status = status_bar
        self._build_ui()

    def _build_ui(self):
        self.tabs = ctk.CTkTabview(self, anchor="nw")
        self.tabs.pack(fill="both", expand=True, padx=5, pady=5)
        self.tabs.add("Estructura SUNDDE")
        self.tabs.add("Precio Justo")
        self.tabs.add("Historial")
        self._build_estructura()
        self._build_precio_justo()
        self._build_historial()

    def _get_periodos(self):
        conn = get_connection()
        rows = conn.execute(
            "SELECT anio||'-'||printf('%02d',mes) as p "
            "FROM periodos ORDER BY anio DESC, mes DESC"
        ).fetchall()
        conn.close()
        return [r["p"] for r in rows] if rows else ["2026-02"]

    def _get_productos(self):
        conn = get_connection()
        rows = conn.execute(
            "SELECT codigo||' - '||descripcion as t "
            "FROM productos_terminados WHERE activo=1"
        ).fetchall()
        conn.close()
        return [r["t"] for r in rows] if rows else ["(sin productos)"]

    # === ESTRUCTURA SUNDDE ===
    def _build_estructura(self):
        tab = self.tabs.tab("Estructura SUNDDE")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)

        ctk.CTkLabel(frame,
            text="Estructura de Costos (LOPJ Art. 13-15)",
            font=ctk.CTkFont(size=18, weight="bold"),
            text_color=NAVY).pack(anchor="w", pady=(0, 5))
        ctk.CTkLabel(frame,
            text="Formato requerido por SUNDDE para determinacion "
                 "de precios regulados",
            font=ctk.CTkFont(size=12),
            text_color=TEAL).pack(anchor="w", pady=(0, 10))

        sel = ctk.CTkFrame(frame, fg_color="transparent")
        sel.pack(fill="x", pady=5)
        self.sd_per = LabelCombo(sel, "Periodo",
                                  self._get_periodos())
        self.sd_per.pack(side="left", padx=(0, 10))
        self.sd_prod = LabelCombo(sel, "Producto",
                                   self._get_productos())
        self.sd_prod.pack(side="left", padx=(0, 10))
        ctk.CTkButton(sel, text="Generar Estructura",
                      fg_color=GREEN, width=180,
                      command=self._gen_estructura).pack(side="left")

        self.sd_frame = ctk.CTkScrollableFrame(frame,
                                                fg_color="transparent")
        self.sd_frame.pack(fill="both", expand=True, pady=10)

    def _gen_estructura(self):
        for w in self.sd_frame.winfo_children():
            w.destroy()
        ps = self.sd_per.get()
        pt = self.sd_prod.get()
        if pt.startswith("("):
            return
        try:
            a, m = int(ps.split("-")[0]), int(ps.split("-")[1])
        except:
            return
        pc = pt.split(" - ")[0].strip()

        conn = get_connection()
        per = conn.execute(
            "SELECT id FROM periodos WHERE anio=? AND mes=?",
            (a, m)).fetchone()
        if not per:
            conn.close()
            return
        pid = per["id"]
        prod = conn.execute(
            "SELECT * FROM productos_terminados WHERE codigo=?",
            (pc,)).fetchone()
        if not prod:
            conn.close()
            return
        prod_id = prod["id"]
        emp = conn.execute(
            "SELECT * FROM empresa LIMIT 1").fetchone()
        empresa = emp["razon_social"] if emp else ""
        rif = emp["rif"] if emp else ""

        # Get order for this product/period
        orden = conn.execute("""
            SELECT * FROM ordenes_produccion
            WHERE producto_id=? AND periodo_id=?
            ORDER BY id DESC LIMIT 1
        """, (prod_id, pid)).fetchone()
        cant = orden["cantidad"] if orden else 1
        oid = orden["id"] if orden else None

        # Calculos
        # MD
        if oid:
            md_total = conn.execute("""
                SELECT COALESCE(SUM(costo_total),0) as t
                FROM movimientos_inventario
                WHERE tipo='salida' AND nota_orden=?
            """, (orden["numero"],)).fetchone()["t"]
        else:
            bom = conn.execute("""
                SELECT SUM(b.cantidad_estandar * m.precio_unitario_bs) as t
                FROM bom b JOIN materiales m ON b.material_id=m.id
                WHERE b.producto_id=?
            """, (prod_id,)).fetchone()
            md_total = (bom["t"] or 0) * cant

        # MOD
        if oid:
            mod_total = conn.execute("""
                SELECT COALESCE(SUM(costo_total),0) as t
                FROM boletas_trabajo WHERE orden_id=?
            """, (oid,)).fetchone()["t"]
        else:
            mod_total = conn.execute("""
                SELECT COALESCE(SUM(n.costo_total_mo),0) as t
                FROM nomina_costos n JOIN trabajadores t
                ON n.trabajador_id=t.id
                WHERE n.periodo_id=? AND t.tipo_mo='MOD'
            """, (pid,)).fetchone()["t"]

        # CIF
        cif_total = conn.execute("""
            SELECT COALESCE(SUM(monto_real),0) as t
            FROM cif_reales WHERE periodo_id=?
        """, (pid,)).fetchone()["t"]

        # MI
        mi_total = conn.execute("""
            SELECT COALESCE(SUM(mi.costo_total),0) as t
            FROM movimientos_inventario mi
            JOIN materiales m ON mi.material_id=m.id
            WHERE mi.tipo='salida' AND mi.fecha LIKE ?
            AND m.tipo='indirecto'
        """, (f"{a}-{m:02d}%",)).fetchone()["t"]

        # MOI
        moi_total = conn.execute("""
            SELECT COALESCE(SUM(n.costo_total_mo),0) as t
            FROM nomina_costos n JOIN trabajadores t
            ON n.trabajador_id=t.id
            WHERE n.periodo_id=? AND t.tipo_mo='MOI'
        """, (pid,)).fetchone()["t"]

        margen_max = get_param("margen_max_lopj") or 30

        conn.close()

        # Calculos finales
        costo_prod = md_total + mod_total + cif_total + mi_total + moi_total
        costo_unit = costo_prod / cant if cant > 0 else 0
        ganancia_max = costo_unit * (margen_max / 100)
        pvp_max = costo_unit + ganancia_max

        # Render estructura
        def ln(lbl, val, b=False, c="333333", indent=0):
            f = ctk.CTkFrame(self.sd_frame, fg_color="transparent")
            f.pack(fill="x", pady=1)
            pref = "  " * indent
            ctk.CTkLabel(f, text=pref + lbl, width=350, anchor="w",
                font=ctk.CTkFont(size=12,
                    weight="bold" if b else "normal"),
                text_color=c).pack(side="left")
            ctk.CTkLabel(f, text=val, anchor="e",
                font=ctk.CTkFont(size=12,
                    weight="bold" if b else "normal"),
                text_color=c).pack(side="right")

        # Header
        ctk.CTkLabel(self.sd_frame,
            text="ESTRUCTURA DE COSTOS SUNDDE",
            font=ctk.CTkFont(size=16, weight="bold"),
            text_color=NAVY).pack(anchor="w")
        ln("Empresa", f"{rif} - {empresa}")
        ln("Producto", f"{prod['codigo']} - {prod['descripcion']}")
        ln("Periodo", ps)
        ln("Unidades Producidas", fmt_num(cant))

        ctk.CTkFrame(self.sd_frame, height=2,
                     fg_color=NAVY).pack(fill="x", pady=8)

        # Section 1: Materia Prima y Materiales
        ctk.CTkLabel(self.sd_frame,
            text="I. MATERIA PRIMA Y MATERIALES",
            font=ctk.CTkFont(size=13, weight="bold"),
            text_color=GREEN).pack(anchor="w", pady=(5, 3))
        ln("Materiales Directos", fmt_bs(md_total), False, "333", 1)
        ln("Materiales Indirectos", fmt_bs(mi_total), False, "333", 1)
        sub_mat = md_total + mi_total
        ln("Subtotal Materiales", fmt_bs(sub_mat), True, GREEN)

        # Section 2: Mano de Obra
        ctk.CTkLabel(self.sd_frame,
            text="II. MANO DE OBRA",
            font=ctk.CTkFont(size=13, weight="bold"),
            text_color=BLUE).pack(anchor="w", pady=(8, 3))
        ln("Mano de Obra Directa", fmt_bs(mod_total), False, "333", 1)
        ln("Mano de Obra Indirecta", fmt_bs(moi_total), False, "333", 1)
        sub_mo = mod_total + moi_total
        ln("Subtotal Mano de Obra", fmt_bs(sub_mo), True, BLUE)

        # Section 3: Gastos de Fabricacion
        ctk.CTkLabel(self.sd_frame,
            text="III. GASTOS DE FABRICACION",
            font=ctk.CTkFont(size=13, weight="bold"),
            text_color=ORANGE).pack(anchor="w", pady=(8, 3))
        ln("CIF Reales", fmt_bs(cif_total), False, "333", 1)
        ln("Subtotal Gastos Fab.", fmt_bs(cif_total), True, ORANGE)

        # Section 4: Costo de Produccion
        ctk.CTkFrame(self.sd_frame, height=3,
                     fg_color=NAVY).pack(fill="x", pady=8)
        ln("IV. COSTO DE PRODUCCION TOTAL",
           fmt_bs(costo_prod), True, NAVY)
        ln("COSTO UNITARIO DE PRODUCCION",
           fmt_bs(costo_unit), True, NAVY)

        # Section 5: Margen y PVP
        ctk.CTkFrame(self.sd_frame, height=2,
                     fg_color=TEAL).pack(fill="x", pady=8)
        ctk.CTkLabel(self.sd_frame,
            text=f"V. MARGEN DE GANANCIA (Max {margen_max:.0f}% - LOPJ Art.15)",
            font=ctk.CTkFont(size=13, weight="bold"),
            text_color=TEAL).pack(anchor="w", pady=(5, 3))
        ln(f"Ganancia Maxima ({margen_max:.0f}%)",
           fmt_bs(ganancia_max), True, TEAL)
        ln("PRECIO VENTA PUBLICO MAXIMO",
           fmt_bs(pvp_max), True, RED)

        if self.status:
            self.status.set_status(
                f"Estructura SUNDDE: PVP max {fmt_bs(pvp_max)}")

    # === PRECIO JUSTO ===
    def _build_precio_justo(self):
        tab = self.tabs.tab("Precio Justo")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)

        ctk.CTkLabel(frame,
            text="Simulador de Precio Justo",
            font=ctk.CTkFont(size=18, weight="bold"),
            text_color=NAVY).pack(anchor="w", pady=(0, 5))
        ctk.CTkLabel(frame,
            text="Calcule el margen real y verifique cumplimiento LOPJ",
            font=ctk.CTkFont(size=12),
            text_color=TEAL).pack(anchor="w", pady=(0, 10))

        form = ctk.CTkFrame(frame, fg_color="transparent")
        form.pack(fill="x")
        self.pj_costo = LabelEntry(form, "Costo Unitario Bs",
                                    placeholder="0.00")
        self.pj_costo.pack(fill="x", pady=3)
        self.pj_pvp = LabelEntry(form, "PVP Deseado Bs",
                                  placeholder="0.00")
        self.pj_pvp.pack(fill="x", pady=3)
        ctk.CTkButton(form, text="Verificar",
                      fg_color=BLUE, width=140,
                      command=self._verificar_pj).pack(pady=10, anchor="w")

        self.pj_result = ctk.CTkLabel(frame, text="",
            font=ctk.CTkFont(size=14), text_color=GRAY,
            wraplength=500, justify="left")
        self.pj_result.pack(anchor="w", pady=10)

    def _verificar_pj(self):
        try:
            costo = float(self.pj_costo.get() or 0)
            pvp = float(self.pj_pvp.get() or 0)
        except:
            self.pj_result.configure(
                text="Ingrese valores numericos.", text_color=RED)
            return
        if costo <= 0:
            self.pj_result.configure(
                text="Costo debe ser mayor a 0.", text_color=RED)
            return

        margen_max = get_param("margen_max_lopj") or 30
        margen_real = ((pvp - costo) / costo) * 100
        pvp_max = costo * (1 + margen_max / 100)
        cumple = margen_real <= margen_max

        color = GREEN if cumple else RED
        status = "CUMPLE" if cumple else "NO CUMPLE"

        txt = (f"Costo Unitario: {fmt_bs(costo)}\n"
               f"PVP Deseado: {fmt_bs(pvp)}\n"
               f"Margen Real: {margen_real:.2f}%\n"
               f"Margen Maximo LOPJ: {margen_max:.0f}%\n"
               f"PVP Maximo Permitido: {fmt_bs(pvp_max)}\n\n"
               f"Estado: {status}")

        self.pj_result.configure(text=txt, text_color=color)

    # === HISTORIAL ===
    def _build_historial(self):
        tab = self.tabs.tab("Historial")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)
        ctk.CTkLabel(frame,
            text="Historial de Estructuras de Costos",
            font=ctk.CTkFont(size=18, weight="bold"),
            text_color=NAVY).pack(anchor="w", pady=(0, 10))
        ctk.CTkLabel(frame,
            text="Las estructuras generadas se almacenan automaticamente\n"
                 "al usar el boton 'Generar Estructura' en la primera pestana.\n"
                 "Proximamente: exportacion a PDF/Excel.",
            font=ctk.CTkFont(size=12), text_color=GRAY,
            justify="left").pack(anchor="w", pady=20)
