"""
Akemi Cost Suite - M7: Reajuste por Inflacion (Fiscal)
Ley ISLR Art. 173-193, Reglamento,
INPC/IPC, Activos Fijos, Inventarios,
Patrimonio Fiscal, Form DPJ-99026
"""
import customtkinter as ctk
from utils.database import get_connection
from utils.common import *


class M7Reajuste(ctk.CTkFrame):
    def __init__(self, master, status_bar=None):
        super().__init__(master, fg_color="transparent")
        self.status = status_bar
        self._build_ui()

    def _build_ui(self):
        self.tabs = ctk.CTkTabview(self, anchor="nw")
        self.tabs.pack(fill="both", expand=True, padx=5, pady=5)
        self.tabs.add("Indices INPC")
        self.tabs.add("Activos Fijos")
        self.tabs.add("Reajuste Ejercicio")
        self.tabs.add("Patrimonio Fiscal")
        self._build_inpc()
        self._build_activos()
        self._build_reajuste()
        self._build_patrimonio()

    def _get_periodos(self):
        conn = get_connection()
        rows = conn.execute(
            "SELECT anio||'-'||printf('%02d',mes) as p "
            "FROM periodos ORDER BY anio DESC, mes DESC"
        ).fetchall()
        conn.close()
        return [r["p"] for r in rows] if rows else ["2026-02"]

    # === INDICES INPC ===
    def _build_inpc(self):
        tab = self.tabs.tab("Indices INPC")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)
        ctk.CTkLabel(frame,
            text="Indices Nacionales de Precios al Consumidor (INPC)",
            font=ctk.CTkFont(size=18, weight="bold"),
            text_color=NAVY).pack(anchor="w", pady=(0, 5))
        ctk.CTkLabel(frame,
            text="BCV / Fuente oficial para calculo de factores de reajuste",
            font=ctk.CTkFont(size=12),
            text_color=TEAL).pack(anchor="w", pady=(0, 10))

        form = ctk.CTkFrame(frame, fg_color="transparent")
        form.pack(fill="x")
        self.in_anio = LabelEntry(form, "Anio", placeholder="2026")
        self.in_anio.pack(fill="x", pady=3)
        self.in_mes = LabelCombo(form, "Mes",
            [str(i) for i in range(1, 13)])
        self.in_mes.pack(fill="x", pady=3)
        self.in_valor = LabelEntry(form, "Valor INPC",
                                    placeholder="0.00")
        self.in_valor.pack(fill="x", pady=3)
        self.in_fuente = LabelEntry(form, "Fuente",
                                     placeholder="BCV")
        self.in_fuente.set("BCV")
        self.in_fuente.pack(fill="x", pady=3)

        ctk.CTkButton(form, text="Registrar INPC", fg_color=GREEN,
                      width=160,
                      command=self._save_inpc).pack(pady=10, anchor="w")

        self.in_table = ScrollableTable(frame,
            columns=["Anio", "Mes", "INPC", "Fuente", "Registrado"],
            widths=[60, 40, 100, 80, 100], height=280)
        self.in_table.pack(fill="both", expand=True)
        self._load_inpc()

    def _load_inpc(self):
        self.in_table.clear_rows()
        conn = get_connection()
        try:
            rows = conn.execute(
                "SELECT * FROM inpc_indices ORDER BY anio DESC, mes DESC"
            ).fetchall()
            for r in rows:
                self.in_table.add_row([
                    r["anio"], r["mes"], fmt_num(r["valor"], 4),
                    r["fuente"], r["fecha_registro"]])
        except:
            pass
        conn.close()

    def _save_inpc(self):
        try:
            anio = int(self.in_anio.get())
            mes = int(self.in_mes.get())
            valor = float(self.in_valor.get())
        except:
            show_message(self, "Error",
                         "Anio, Mes y Valor requeridos.", "error")
            return
        fuente = self.in_fuente.get() or "BCV"
        conn = get_connection()
        try:
            conn.execute("""INSERT INTO inpc_indices
                (anio, mes, valor, fuente, fecha_registro)
                VALUES (?,?,?,?,?)""",
                (anio, mes, valor, fuente, fecha_hoy()))
            conn.commit()
        except Exception as e:
            show_message(self, "Error", str(e), "error")
        conn.close()
        self.in_valor.set("")
        self._load_inpc()

    # === ACTIVOS FIJOS ===
    def _build_activos(self):
        tab = self.tabs.tab("Activos Fijos")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)
        ctk.CTkLabel(frame,
            text="Activos Fijos Depreciables (Art. 179 LISLR)",
            font=ctk.CTkFont(size=18, weight="bold"),
            text_color=NAVY).pack(anchor="w", pady=(0, 10))

        form = ctk.CTkFrame(frame, fg_color="transparent")
        form.pack(fill="x")
        self.af_rubro = LabelEntry(form, "Rubro/Categoria",
                                    placeholder="Maquinaria, Vehiculos...")
        self.af_rubro.pack(fill="x", pady=3)
        self.af_desc = LabelEntry(form, "Descripcion",
                                   placeholder="Prensa Offset modelo X")
        self.af_desc.pack(fill="x", pady=3)
        self.af_fecha = LabelEntry(form, "Fecha Adquisicion",
                                    placeholder="2020-01-15")
        self.af_fecha.pack(fill="x", pady=3)
        self.af_vida = LabelEntry(form, "Vida Util (meses)",
                                   placeholder="120")
        self.af_vida.pack(fill="x", pady=3)
        self.af_costo = LabelEntry(form, "Costo Historico Bs",
                                    placeholder="0.00")
        self.af_costo.pack(fill="x", pady=3)
        self.af_dep = LabelEntry(form, "Depreciacion Acum. Bs",
                                  placeholder="0.00")
        self.af_dep.pack(fill="x", pady=3)
        self.af_inpc = LabelEntry(form, "INPC Fecha Origen",
                                   placeholder="0.00")
        self.af_inpc.pack(fill="x", pady=3)

        bf = ctk.CTkFrame(frame, fg_color="transparent")
        bf.pack(fill="x", pady=8)
        ctk.CTkButton(bf, text="Guardar", fg_color=GREEN,
                      width=120,
                      command=self._save_af).pack(side="left", padx=5)
        ctk.CTkButton(bf, text="Limpiar", fg_color=GRAY,
                      width=100,
                      command=self._clear_af).pack(side="left", padx=5)
        self.af_eid = None

        self.af_table = ScrollableTable(frame,
            columns=["ID", "Rubro", "Descripcion", "Costo Hist.",
                     "Dep.Acum.", "Neto"],
            widths=[35, 100, 180, 80, 80, 80], height=200)
        self.af_table.pack(fill="both", expand=True)
        self._load_afs()

    def _load_afs(self):
        self.af_table.clear_rows()
        conn = get_connection()
        try:
            rows = conn.execute(
                "SELECT * FROM activos_fijos ORDER BY rubro"
            ).fetchall()
            for r in rows:
                neto = r["costo_historico"] - r["depreciacion_acum"]
                self.af_table.add_row([
                    r["id"], r["rubro"], r["descripcion"][:25],
                    fmt_num(r["costo_historico"]),
                    fmt_num(r["depreciacion_acum"]),
                    fmt_num(neto)
                ], tag=r["id"], on_click=self._sel_af)
        except:
            pass
        conn.close()

    def _sel_af(self, aid):
        conn = get_connection()
        r = conn.execute("SELECT * FROM activos_fijos WHERE id=?",
                         (aid,)).fetchone()
        conn.close()
        if not r:
            return
        self.af_eid = r["id"]
        self.af_rubro.set(r["rubro"])
        self.af_desc.set(r["descripcion"])
        self.af_fecha.set(r["fecha_origen"])
        self.af_vida.set(r["vida_util_meses"])
        self.af_costo.set(r["costo_historico"])
        self.af_dep.set(r["depreciacion_acum"])
        self.af_inpc.set(r["inpc_origen"])

    def _save_af(self):
        rubro = self.af_rubro.get()
        desc = self.af_desc.get()
        if not rubro or not desc:
            show_message(self, "Error",
                         "Rubro y Descripcion requeridos.", "error")
            return
        fecha = self.af_fecha.get()
        try:
            vida = int(self.af_vida.get() or 120)
        except:
            vida = 120
        try:
            costo = float(self.af_costo.get() or 0)
        except:
            costo = 0
        try:
            dep = float(self.af_dep.get() or 0)
        except:
            dep = 0
        try:
            inpc = float(self.af_inpc.get() or 0)
        except:
            inpc = 0

        conn = get_connection()
        try:
            if self.af_eid:
                conn.execute("""UPDATE activos_fijos SET rubro=?,
                    descripcion=?, fecha_origen=?, vida_util_meses=?,
                    costo_historico=?, depreciacion_acum=?,
                    inpc_origen=? WHERE id=?""",
                    (rubro, desc, fecha, vida, costo, dep, inpc,
                     self.af_eid))
            else:
                conn.execute("""INSERT INTO activos_fijos
                    (rubro, descripcion, fecha_origen, vida_util_meses,
                     costo_historico, depreciacion_acum, inpc_origen)
                    VALUES (?,?,?,?,?,?,?)""",
                    (rubro, desc, fecha, vida, costo, dep, inpc))
            conn.commit()
        except Exception as e:
            show_message(self, "Error", str(e), "error")
        conn.close()
        self._clear_af()
        self._load_afs()

    def _clear_af(self):
        self.af_eid = None
        for w in [self.af_rubro, self.af_desc, self.af_fecha,
                  self.af_vida, self.af_costo, self.af_dep,
                  self.af_inpc]:
            w.set("")

    # === REAJUSTE DEL EJERCICIO ===
    def _build_reajuste(self):
        tab = self.tabs.tab("Reajuste Ejercicio")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)
        ctk.CTkLabel(frame,
            text="Reajuste Regular por Inflacion (Art. 178 LISLR)",
            font=ctk.CTkFont(size=18, weight="bold"),
            text_color=NAVY).pack(anchor="w", pady=(0, 5))
        ctk.CTkLabel(frame,
            text="Calculo automatico de factores INPC y reajuste neto",
            font=ctk.CTkFont(size=12),
            text_color=TEAL).pack(anchor="w", pady=(0, 10))

        sel = ctk.CTkFrame(frame, fg_color="transparent")
        sel.pack(fill="x", pady=5)
        self.ra_per = LabelCombo(sel, "Periodo Fiscal",
                                  self._get_periodos())
        self.ra_per.pack(side="left", padx=(0, 10))
        ctk.CTkButton(sel, text="Calcular Reajuste Activos Fijos",
                      fg_color=BLUE, width=240,
                      command=self._calc_reajuste_af).pack(side="left")

        self.ra_frame = ctk.CTkScrollableFrame(frame,
                                                fg_color="transparent")
        self.ra_frame.pack(fill="both", expand=True, pady=10)

    def _calc_reajuste_af(self):
        for w in self.ra_frame.winfo_children():
            w.destroy()
        ps = self.ra_per.get()
        try:
            a, m = int(ps.split("-")[0]), int(ps.split("-")[1])
        except:
            return

        conn = get_connection()
        # Get INPC del cierre
        inpc_cierre = None
        try:
            ic = conn.execute(
                "SELECT valor FROM inpc_indices WHERE anio=? AND mes=?",
                (a, m)).fetchone()
            if ic:
                inpc_cierre = ic["valor"]
        except:
            pass

        if not inpc_cierre:
            ctk.CTkLabel(self.ra_frame,
                text=f"Falta INPC para {ps}. Registrelo en la pestana Indices.",
                text_color=RED).pack(pady=20)
            conn.close()
            return

        activos = []
        try:
            activos = conn.execute(
                "SELECT * FROM activos_fijos ORDER BY rubro"
            ).fetchall()
        except:
            pass

        if not activos:
            ctk.CTkLabel(self.ra_frame,
                text="Sin activos fijos registrados.",
                text_color=ORANGE).pack(pady=20)
            conn.close()
            return

        ctk.CTkLabel(self.ra_frame,
            text=f"Reajuste al {ps} | INPC Cierre: {fmt_num(inpc_cierre,4)}",
            font=ctk.CTkFont(size=14, weight="bold"),
            text_color=NAVY).pack(anchor="w", pady=(0, 10))

        def ln(lbl, val, b=False, c="333333"):
            f = ctk.CTkFrame(self.ra_frame, fg_color="transparent")
            f.pack(fill="x", pady=1)
            ctk.CTkLabel(f, text=lbl, width=320, anchor="w",
                font=ctk.CTkFont(size=12,
                    weight="bold" if b else "normal"),
                text_color=c).pack(side="left")
            ctk.CTkLabel(f, text=val, anchor="e",
                font=ctk.CTkFont(size=12,
                    weight="bold" if b else "normal"),
                text_color=c).pack(side="right")

        total_reajuste = 0
        per = conn.execute(
            "SELECT id FROM periodos WHERE anio=? AND mes=?",
            (a, m)).fetchone()
        pid = per["id"] if per else None

        # Clear previous
        if pid:
            try:
                conn.execute(
                    "DELETE FROM reajuste_ejercicio WHERE periodo_id=?",
                    (pid,))
            except:
                pass

        for af in activos:
            inpc_orig = af["inpc_origen"] or 1
            if inpc_orig <= 0:
                inpc_orig = 1
            factor = inpc_cierre / inpc_orig
            neto_hist = af["costo_historico"] - af["depreciacion_acum"]
            neto_ajust = neto_hist * factor
            reajuste = neto_ajust - neto_hist

            ctk.CTkFrame(self.ra_frame, height=1,
                         fg_color=GRAY).pack(fill="x", pady=4)
            ctk.CTkLabel(self.ra_frame,
                text=f"{af['rubro']}: {af['descripcion']}",
                font=ctk.CTkFont(size=12, weight="bold"),
                text_color=BLUE).pack(anchor="w")
            ln("  Costo Historico", fmt_bs(af["costo_historico"]))
            ln("  Dep. Acumulada", fmt_bs(af["depreciacion_acum"]))
            ln("  Valor Neto Hist.", fmt_bs(neto_hist))
            ln(f"  Factor (INPC {fmt_num(inpc_cierre,2)}/{fmt_num(inpc_orig,2)})",
               fmt_num(factor, 6))
            ln("  Valor Neto Ajustado", fmt_bs(neto_ajust))
            ln("  REAJUSTE NETO", fmt_bs(reajuste), True,
               GREEN if reajuste >= 0 else RED)

            total_reajuste += reajuste

            # Save
            if pid:
                try:
                    conn.execute("""INSERT INTO reajuste_ejercicio
                        (periodo_id, tipo, descripcion, valor_historico,
                         factor, valor_actualizado, reajuste_neto)
                        VALUES (?,?,?,?,?,?,?)""",
                        (pid, "activo_fijo",
                         f"{af['rubro']} - {af['descripcion']}",
                         neto_hist, factor, neto_ajust, reajuste))
                except:
                    pass

        conn.commit()
        conn.close()

        ctk.CTkFrame(self.ra_frame, height=3,
                     fg_color=NAVY).pack(fill="x", pady=10)
        ln("TOTAL REAJUSTE ACTIVOS FIJOS",
           fmt_bs(total_reajuste), True, NAVY)

        if self.status:
            self.status.set_status(
                f"Reajuste calculado: {fmt_bs(total_reajuste)}")

    # === PATRIMONIO FISCAL ===
    def _build_patrimonio(self):
        tab = self.tabs.tab("Patrimonio Fiscal")
        frame = ctk.CTkFrame(tab, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=10, pady=5)
        ctk.CTkLabel(frame,
            text="Patrimonio Neto Fiscal (Art. 184 LISLR)",
            font=ctk.CTkFont(size=18, weight="bold"),
            text_color=NAVY).pack(anchor="w", pady=(0, 5))
        ctk.CTkLabel(frame,
            text="Activos No Monetarios - Pasivos No Monetarios = "
                 "Patrimonio Neto",
            font=ctk.CTkFont(size=12),
            text_color=TEAL).pack(anchor="w", pady=(0, 10))

        form = ctk.CTkFrame(frame, fg_color="transparent")
        form.pack(fill="x")
        self.pf_per = LabelCombo(form, "Periodo",
                                  self._get_periodos())
        self.pf_per.pack(fill="x", pady=3)
        self.pf_concepto = LabelEntry(form, "Concepto",
            placeholder="Capital Social, Reserva Legal...")
        self.pf_concepto.pack(fill="x", pady=3)
        self.pf_tipo = LabelCombo(form, "Tipo",
            ["activo_no_monetario", "pasivo_no_monetario",
             "patrimonio"])
        self.pf_tipo.pack(fill="x", pady=3)
        self.pf_hist = LabelEntry(form, "Monto Historico Bs",
                                   placeholder="0.00")
        self.pf_hist.pack(fill="x", pady=3)
        self.pf_act = LabelEntry(form, "Monto Actualizado Bs",
                                  placeholder="0.00")
        self.pf_act.pack(fill="x", pady=3)

        ctk.CTkButton(form, text="Registrar", fg_color=GREEN,
                      width=140,
                      command=self._save_pf).pack(pady=10, anchor="w")

        self.pf_table = ScrollableTable(frame,
            columns=["Concepto", "Tipo", "Historico",
                     "Actualizado", "Diferencia"],
            widths=[150, 120, 80, 80, 80], height=220)
        self.pf_table.pack(fill="both", expand=True)
        self._load_pf()

    def _save_pf(self):
        ps = self.pf_per.get()
        try:
            a, m = int(ps.split("-")[0]), int(ps.split("-")[1])
        except:
            return
        concepto = self.pf_concepto.get()
        if not concepto:
            return
        tipo = self.pf_tipo.get()
        try:
            hist = float(self.pf_hist.get() or 0)
        except:
            hist = 0
        try:
            act = float(self.pf_act.get() or 0)
        except:
            act = 0

        conn = get_connection()
        per = conn.execute(
            "SELECT id FROM periodos WHERE anio=? AND mes=?",
            (a, m)).fetchone()
        if not per:
            conn.close()
            return
        try:
            conn.execute("""INSERT INTO patrimonio_fiscal
                (periodo_id, concepto, monto_historico,
                 monto_actualizado, tipo, fecha)
                VALUES (?,?,?,?,?,?)""",
                (per["id"], concepto, hist, act, tipo, fecha_hoy()))
            conn.commit()
        except Exception as e:
            show_message(self, "Error", str(e), "error")
        conn.close()
        self.pf_concepto.set("")
        self.pf_hist.set("")
        self.pf_act.set("")
        self._load_pf()

    def _load_pf(self):
        self.pf_table.clear_rows()
        conn = get_connection()
        try:
            rows = conn.execute("""
                SELECT * FROM patrimonio_fiscal
                ORDER BY id DESC LIMIT 50
            """).fetchall()
            for r in rows:
                dif = r["monto_actualizado"] - r["monto_historico"]
                self.pf_table.add_row([
                    r["concepto"][:20], r["tipo"],
                    fmt_num(r["monto_historico"]),
                    fmt_num(r["monto_actualizado"]),
                    fmt_num(dif)])
        except:
            pass
        conn.close()
