"""
Akemi Cost Suite - Utilidades comunes
Colores, formateo, widgets reutilizables
"""
import customtkinter as ctk
from datetime import datetime

# === COLORES CORPORATIVOS ===
NAVY = "#1B3A5C"
BLUE = "#2E75B6"
LIGHT_BLUE = "#D6E4F0"
VERY_LIGHT = "#EDF2F9"
WHITE = "#FFFFFF"
GREEN = "#2E7D32"
RED = "#C62828"
ORANGE = "#E65100"
TEAL = "#00796B"
GRAY = "#666666"
LIGHT_GRAY = "#F5F5F5"
DARK = "#1a1a2e"
CARD_BG = "#2d2d44"
TEXT_PRIMARY = "#E0E0E0"
TEXT_SECONDARY = "#A0A0A0"
ACCENT = "#4FC3F7"

# === FORMATO ===
def fmt_bs(valor):
    """Formato bolivares: 1,234.56 Bs"""
    if valor is None:
        return "0.00 Bs"
    return f"{valor:,.2f} Bs"

def fmt_usd(valor):
    """Formato dolares: $1,234.56"""
    if valor is None:
        return "$0.00"
    return f"${valor:,.2f}"

def fmt_num(valor, decimales=2):
    """Formato numerico general"""
    if valor is None:
        return "0"
    return f"{valor:,.{decimales}f}"

def fmt_fecha(fecha_str):
    """Formato fecha legible"""
    if not fecha_str:
        return ""
    try:
        dt = datetime.strptime(fecha_str[:10], "%Y-%m-%d")
        return dt.strftime("%d/%m/%Y")
    except Exception:
        return fecha_str

def fecha_hoy():
    return datetime.now().strftime("%Y-%m-%d")

def fecha_hora():
    return datetime.now().strftime("%Y-%m-%d %H:%M:%S")


# === WIDGETS REUTILIZABLES ===

class LabelEntry(ctk.CTkFrame):
    """Label + Entry en una fila"""
    def __init__(self, master, label, width=200, placeholder="", **kwargs):
        super().__init__(master, fg_color="transparent")
        self.lbl = ctk.CTkLabel(self, text=label, width=160, anchor="w",
                                font=ctk.CTkFont(size=13))
        self.lbl.pack(side="left", padx=(0, 8))
        self.entry = ctk.CTkEntry(self, width=width, placeholder_text=placeholder, **kwargs)
        self.entry.pack(side="left", fill="x", expand=True)

    def get(self):
        return self.entry.get().strip()

    def set(self, value):
        self.entry.delete(0, "end")
        self.entry.insert(0, str(value) if value else "")

    def configure_entry(self, **kwargs):
        self.entry.configure(**kwargs)


class LabelCombo(ctk.CTkFrame):
    """Label + ComboBox en una fila"""
    def __init__(self, master, label, values, width=200, **kwargs):
        super().__init__(master, fg_color="transparent")
        self.lbl = ctk.CTkLabel(self, text=label, width=160, anchor="w",
                                font=ctk.CTkFont(size=13))
        self.lbl.pack(side="left", padx=(0, 8))
        self.combo = ctk.CTkComboBox(self, values=values, width=width, **kwargs)
        self.combo.pack(side="left")

    def get(self):
        return self.combo.get()

    def set(self, value):
        self.combo.set(str(value) if value else "")


class StatusBar(ctk.CTkFrame):
    """Barra de estado inferior"""
    def __init__(self, master):
        super().__init__(master, height=30, fg_color=NAVY)
        self.pack_propagate(False)
        self.label = ctk.CTkLabel(self, text="Listo", font=ctk.CTkFont(size=11),
                                  text_color="#CCCCCC")
        self.label.pack(side="left", padx=10)
        self.right = ctk.CTkLabel(self, text="", font=ctk.CTkFont(size=11),
                                  text_color="#CCCCCC")
        self.right.pack(side="right", padx=10)

    def set_status(self, text):
        self.label.configure(text=text)

    def set_right(self, text):
        self.right.configure(text=text)


class ScrollableTable(ctk.CTkScrollableFrame):
    """Tabla scrollable simple con headers y filas"""
    def __init__(self, master, columns, widths=None, **kwargs):
        super().__init__(master, **kwargs)
        self.columns = columns
        self.widths = widths or [120] * len(columns)
        self.rows_data = []
        self.row_frames = []
        self._build_header()

    def _build_header(self):
        hdr = ctk.CTkFrame(self, fg_color=NAVY, corner_radius=0)
        hdr.pack(fill="x", pady=(0, 2))
        for i, col in enumerate(self.columns):
            lbl = ctk.CTkLabel(hdr, text=col, width=self.widths[i],
                               font=ctk.CTkFont(size=12, weight="bold"),
                               text_color="white")
            lbl.pack(side="left", padx=2, pady=4)

    def clear_rows(self):
        for f in self.row_frames:
            f.destroy()
        self.row_frames = []
        self.rows_data = []

    def add_row(self, values, tag=None, on_click=None):
        idx = len(self.row_frames)
        bg = VERY_LIGHT if idx % 2 == 0 else WHITE
        row_frame = ctk.CTkFrame(self, fg_color=bg, corner_radius=0)
        row_frame.pack(fill="x", pady=1)
        for i, val in enumerate(values):
            w = self.widths[i] if i < len(self.widths) else 120
            lbl = ctk.CTkLabel(row_frame, text=str(val), width=w,
                               font=ctk.CTkFont(size=12), anchor="w")
            lbl.pack(side="left", padx=2, pady=3)
            if on_click:
                lbl.bind("<Button-1>", lambda e, t=tag: on_click(t))
        self.row_frames.append(row_frame)
        self.rows_data.append({"values": values, "tag": tag})


def show_message(master, title, message, msg_type="info"):
    """Muestra un dialogo de mensaje"""
    dialog = ctk.CTkToplevel(master)
    dialog.title(title)
    dialog.geometry("400x180")
    dialog.resizable(False, False)
    dialog.transient(master)
    dialog.grab_set()

    color = GREEN if msg_type == "success" else (RED if msg_type == "error" else BLUE)

    frame = ctk.CTkFrame(dialog, fg_color="transparent")
    frame.pack(fill="both", expand=True, padx=20, pady=20)

    ctk.CTkLabel(frame, text=title, font=ctk.CTkFont(size=16, weight="bold"),
                 text_color=color).pack(pady=(0, 10))
    ctk.CTkLabel(frame, text=message, font=ctk.CTkFont(size=13),
                 wraplength=340).pack(pady=(0, 15))
    ctk.CTkButton(frame, text="Aceptar", width=100, fg_color=color,
                  command=dialog.destroy).pack()

    dialog.after(100, lambda: dialog.focus_force())


def confirm_dialog(master, title, message):
    """Dialogo de confirmacion Si/No. Retorna True si confirma."""
    result = [False]
    dialog = ctk.CTkToplevel(master)
    dialog.title(title)
    dialog.geometry("420x180")
    dialog.resizable(False, False)
    dialog.transient(master)
    dialog.grab_set()

    frame = ctk.CTkFrame(dialog, fg_color="transparent")
    frame.pack(fill="both", expand=True, padx=20, pady=20)

    ctk.CTkLabel(frame, text=message, font=ctk.CTkFont(size=13),
                 wraplength=360).pack(pady=(10, 20))

    btn_frame = ctk.CTkFrame(frame, fg_color="transparent")
    btn_frame.pack()

    def on_yes():
        result[0] = True
        dialog.destroy()

    ctk.CTkButton(btn_frame, text="Si", width=80, fg_color=GREEN,
                  command=on_yes).pack(side="left", padx=10)
    ctk.CTkButton(btn_frame, text="No", width=80, fg_color=RED,
                  command=dialog.destroy).pack(side="left", padx=10)

    dialog.wait_window()
    return result[0]
