"""
Akemi Cost Suite - Motor de Base de Datos
Todas las tablas SQLite + funciones CRUD base
"""
import sqlite3
import os
import shutil
from datetime import datetime, date

DB_DIR = os.path.join(os.path.dirname(os.path.abspath(__file__)), "..", "data")
DB_PATH = os.path.join(DB_DIR, "akemi_cost.db")
BACKUP_DIR = os.path.join(DB_DIR, "backups")


def get_db_path():
    return DB_PATH


def get_connection():
    os.makedirs(DB_DIR, exist_ok=True)
    conn = sqlite3.connect(DB_PATH)
    conn.row_factory = sqlite3.Row
    conn.execute("PRAGMA journal_mode=WAL")
    conn.execute("PRAGMA foreign_keys=ON")
    return conn


def init_db():
    conn = get_connection()
    c = conn.cursor()

    # === M0: CONFIGURACION ===
    c.execute("""CREATE TABLE IF NOT EXISTS empresa (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        rif TEXT NOT NULL DEFAULT '',
        razon_social TEXT NOT NULL DEFAULT 'Mi Empresa',
        direccion TEXT DEFAULT '',
        telefono TEXT DEFAULT '',
        actividad_economica TEXT DEFAULT '',
        clasificacion_sundde TEXT DEFAULT '',
        logo_path TEXT DEFAULT '',
        moneda_base TEXT DEFAULT 'Bs',
        horas_semanales REAL DEFAULT 44,
        created_at TEXT DEFAULT (datetime('now','localtime')),
        updated_at TEXT DEFAULT (datetime('now','localtime'))
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS parametros_legales (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        clave TEXT NOT NULL UNIQUE,
        valor REAL NOT NULL,
        descripcion TEXT DEFAULT '',
        base_legal TEXT DEFAULT '',
        editable INTEGER DEFAULT 1,
        vigente_desde TEXT DEFAULT ''
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS periodos (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        anio INTEGER NOT NULL,
        mes INTEGER NOT NULL,
        estado TEXT DEFAULT 'abierto',
        fecha_cierre TEXT DEFAULT '',
        UNIQUE(anio, mes)
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS tasas_bcv (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        fecha TEXT NOT NULL,
        tasa_usd REAL NOT NULL,
        fuente TEXT DEFAULT 'manual',
        created_at TEXT DEFAULT (datetime('now','localtime'))
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS usuarios (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        nombre TEXT NOT NULL,
        clave_hash TEXT DEFAULT '',
        rol TEXT DEFAULT 'operador',
        activo INTEGER DEFAULT 1
    )""")

    # === M1: MATERIALES E INVENTARIOS ===
    c.execute("""CREATE TABLE IF NOT EXISTS categorias_material (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        nombre TEXT NOT NULL UNIQUE,
        descripcion TEXT DEFAULT ''
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS materiales (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        codigo TEXT NOT NULL UNIQUE,
        descripcion TEXT NOT NULL,
        unidad_medida TEXT NOT NULL DEFAULT 'Und',
        tipo TEXT NOT NULL DEFAULT 'directo',
        categoria_id INTEGER DEFAULT NULL,
        proveedor TEXT DEFAULT '',
        precio_unitario_bs REAL DEFAULT 0,
        precio_unitario_usd REAL DEFAULT 0,
        presentacion_comercial TEXT DEFAULT '',
        cantidad_por_presentacion REAL DEFAULT 1,
        stock_actual REAL DEFAULT 0,
        stock_minimo REAL DEFAULT 0,
        cuenta_contable TEXT DEFAULT '',
        activo INTEGER DEFAULT 1,
        created_at TEXT DEFAULT (datetime('now','localtime')),
        updated_at TEXT DEFAULT (datetime('now','localtime')),
        FOREIGN KEY (categoria_id) REFERENCES categorias_material(id)
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS movimientos_inventario (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        material_id INTEGER NOT NULL,
        tipo TEXT NOT NULL,
        cantidad REAL NOT NULL,
        costo_unitario REAL DEFAULT 0,
        costo_total REAL DEFAULT 0,
        factura TEXT DEFAULT '',
        proveedor TEXT DEFAULT '',
        orden_id INTEGER DEFAULT NULL,
        departamento_id INTEGER DEFAULT NULL,
        periodo_id INTEGER DEFAULT NULL,
        fecha TEXT NOT NULL,
        lote TEXT DEFAULT '',
        nota TEXT DEFAULT '',
        created_at TEXT DEFAULT (datetime('now','localtime')),
        FOREIGN KEY (material_id) REFERENCES materiales(id)
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS kardex (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        material_id INTEGER NOT NULL,
        movimiento_id INTEGER NOT NULL,
        fecha TEXT NOT NULL,
        tipo TEXT NOT NULL,
        cantidad_entrada REAL DEFAULT 0,
        costo_unit_entrada REAL DEFAULT 0,
        total_entrada REAL DEFAULT 0,
        cantidad_salida REAL DEFAULT 0,
        costo_unit_salida REAL DEFAULT 0,
        total_salida REAL DEFAULT 0,
        cantidad_saldo REAL DEFAULT 0,
        costo_unit_saldo REAL DEFAULT 0,
        total_saldo REAL DEFAULT 0,
        FOREIGN KEY (material_id) REFERENCES materiales(id),
        FOREIGN KEY (movimiento_id) REFERENCES movimientos_inventario(id)
    )""")

    # === M2: MANO DE OBRA (estructura lista para Fase 2) ===
    c.execute("""CREATE TABLE IF NOT EXISTS departamentos (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        nombre TEXT NOT NULL UNIQUE,
        tipo TEXT DEFAULT 'produccion',
        capacidad_normal REAL DEFAULT 0,
        es_centro_costos INTEGER DEFAULT 1,
        recibe_costos_fijos INTEGER DEFAULT 0,
        orden_proceso INTEGER DEFAULT 0
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS trabajadores (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        cedula TEXT NOT NULL UNIQUE,
        nombre TEXT NOT NULL,
        cargo TEXT DEFAULT '',
        departamento_id INTEGER DEFAULT NULL,
        tipo_mo TEXT DEFAULT 'MOD',
        fecha_ingreso TEXT DEFAULT '',
        salario_basico_mensual REAL DEFAULT 0,
        riesgo_ivss TEXT DEFAULT 'minimo',
        dias_utilidades INTEGER DEFAULT 30,
        dias_bono_vacacional INTEGER DEFAULT 15,
        activo INTEGER DEFAULT 1,
        created_at TEXT DEFAULT (datetime('now','localtime')),
        FOREIGN KEY (departamento_id) REFERENCES departamentos(id)
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS nomina_costos (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        trabajador_id INTEGER NOT NULL,
        periodo_id INTEGER NOT NULL,
        salario_normal_diario REAL DEFAULT 0,
        alic_utilidades REAL DEFAULT 0,
        alic_bono_vac REAL DEFAULT 0,
        salario_integral REAL DEFAULT 0,
        sso_patronal REAL DEFAULT 0,
        paro_forzoso REAL DEFAULT 0,
        faov REAL DEFAULT 0,
        inces REAL DEFAULT 0,
        cepp REAL DEFAULT 0,
        prov_prestaciones REAL DEFAULT 0,
        prov_utilidades REAL DEFAULT 0,
        prov_bono_vac REAL DEFAULT 0,
        prov_vacaciones REAL DEFAULT 0,
        total_beneficios REAL DEFAULT 0,
        costo_total_mo REAL DEFAULT 0,
        FOREIGN KEY (trabajador_id) REFERENCES trabajadores(id),
        FOREIGN KEY (periodo_id) REFERENCES periodos(id)
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS boletas_trabajo (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        trabajador_id INTEGER NOT NULL,
        orden_id INTEGER DEFAULT NULL,
        departamento_id INTEGER DEFAULT NULL,
        fecha TEXT NOT NULL,
        horas REAL NOT NULL,
        costo_hora REAL DEFAULT 0,
        costo_total REAL DEFAULT 0,
        periodo_id INTEGER DEFAULT NULL,
        FOREIGN KEY (trabajador_id) REFERENCES trabajadores(id)
    )""")

    # === M3: CIF ===
    c.execute("""CREATE TABLE IF NOT EXISTS cif_catalogo (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        descripcion TEXT NOT NULL,
        tipo TEXT DEFAULT 'fijo',
        unidad_medida TEXT DEFAULT '',
        costo_por_unidad REAL DEFAULT 0,
        cuenta_contable TEXT DEFAULT '',
        activo INTEGER DEFAULT 1
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS cif_presupuesto (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        periodo_id INTEGER NOT NULL,
        cif_id INTEGER NOT NULL,
        monto_presupuestado REAL DEFAULT 0,
        base_actividad TEXT DEFAULT 'horas_mod',
        cantidad_base_presup REAL DEFAULT 0,
        FOREIGN KEY (periodo_id) REFERENCES periodos(id),
        FOREIGN KEY (cif_id) REFERENCES cif_catalogo(id)
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS cif_reales (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        periodo_id INTEGER NOT NULL,
        cif_id INTEGER NOT NULL,
        departamento_id INTEGER DEFAULT NULL,
        monto_real REAL DEFAULT 0,
        fecha TEXT DEFAULT '',
        descripcion TEXT DEFAULT '',
        FOREIGN KEY (periodo_id) REFERENCES periodos(id),
        FOREIGN KEY (cif_id) REFERENCES cif_catalogo(id)
    )""")

    # === M4: PRODUCCION ===
    c.execute("""CREATE TABLE IF NOT EXISTS productos_terminados (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        codigo TEXT NOT NULL UNIQUE,
        descripcion TEXT NOT NULL,
        unidad TEXT DEFAULT 'Und',
        costo_unitario_actual REAL DEFAULT 0,
        stock REAL DEFAULT 0,
        precio_justo REAL DEFAULT 0,
        margen_aplicado REAL DEFAULT 0,
        activo INTEGER DEFAULT 1,
        created_at TEXT DEFAULT (datetime('now','localtime'))
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS bom (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        producto_id INTEGER NOT NULL,
        material_id INTEGER NOT NULL,
        cantidad_requerida REAL NOT NULL,
        unidad TEXT DEFAULT '',
        operacion TEXT DEFAULT '',
        horas_mo REAL DEFAULT 0,
        horas_maquina REAL DEFAULT 0,
        FOREIGN KEY (producto_id) REFERENCES productos_terminados(id),
        FOREIGN KEY (material_id) REFERENCES materiales(id)
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS ordenes_produccion (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        numero TEXT NOT NULL UNIQUE,
        producto_id INTEGER DEFAULT NULL,
        cantidad REAL DEFAULT 0,
        fecha_inicio TEXT DEFAULT '',
        fecha_fin TEXT DEFAULT '',
        estado TEXT DEFAULT 'abierta',
        cliente TEXT DEFAULT '',
        costo_mp REAL DEFAULT 0,
        costo_mod REAL DEFAULT 0,
        costo_cif REAL DEFAULT 0,
        costo_total REAL DEFAULT 0,
        costo_unitario REAL DEFAULT 0,
        tipo_costeo TEXT DEFAULT 'ordenes',
        periodo_id INTEGER DEFAULT NULL,
        created_at TEXT DEFAULT (datetime('now','localtime')),
        FOREIGN KEY (producto_id) REFERENCES productos_terminados(id)
    )""")

    # M7: Reajuste por Inflacion
    c.execute("""CREATE TABLE IF NOT EXISTS inpc_indices (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        anio INTEGER NOT NULL,
        mes INTEGER NOT NULL,
        valor REAL NOT NULL,
        fuente TEXT DEFAULT 'BCV',
        fecha_registro TEXT
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS activos_fijos (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        rubro TEXT NOT NULL,
        descripcion TEXT NOT NULL,
        fecha_origen TEXT,
        vida_util_meses INTEGER DEFAULT 120,
        costo_historico REAL DEFAULT 0,
        depreciacion_acum REAL DEFAULT 0,
        inpc_origen REAL DEFAULT 0,
        cuenta_contable TEXT DEFAULT ''
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS reajuste_ejercicio (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        periodo_id INTEGER,
        tipo TEXT,
        descripcion TEXT,
        valor_historico REAL DEFAULT 0,
        factor REAL DEFAULT 1,
        valor_actualizado REAL DEFAULT 0,
        reajuste_neto REAL DEFAULT 0,
        FOREIGN KEY (periodo_id) REFERENCES periodos(id)
    )""")

    c.execute("""CREATE TABLE IF NOT EXISTS patrimonio_fiscal (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        periodo_id INTEGER,
        concepto TEXT,
        monto_historico REAL DEFAULT 0,
        monto_actualizado REAL DEFAULT 0,
        tipo TEXT DEFAULT 'patrimonio',
        fecha TEXT,
        inpc_fecha REAL DEFAULT 0,
        FOREIGN KEY (periodo_id) REFERENCES periodos(id)
    )""")

    # Datos por defecto
    _insert_defaults(c)

    # Migraciones: alinear columnas con modulos
    _migrate(c)

    conn.commit()
    conn.close()


def _migrate(c):
    """Agrega columnas faltantes sin perder datos"""
    def add_col(table, col, typedef):
        try:
            c.execute(f"ALTER TABLE {table} ADD COLUMN {col} {typedef}")
        except Exception:
            pass

    # departamentos: M4 espera orden_secuencia, capacidad_normal_hrs, activo
    add_col("departamentos", "orden_secuencia", "INTEGER DEFAULT 0")
    add_col("departamentos", "capacidad_normal_hrs", "REAL DEFAULT 176")
    add_col("departamentos", "activo", "INTEGER DEFAULT 1")
    try:
        c.execute("UPDATE departamentos SET orden_secuencia=orden_proceso WHERE orden_secuencia=0 AND orden_proceso>0")
    except Exception:
        pass
    try:
        c.execute("UPDATE departamentos SET capacidad_normal_hrs=capacidad_normal WHERE capacidad_normal_hrs=176 AND capacidad_normal>0")
    except Exception:
        pass

    # productos_terminados: M4 espera unidad_medida
    add_col("productos_terminados", "unidad_medida", "TEXT DEFAULT 'Und'")
    try:
        c.execute("UPDATE productos_terminados SET unidad_medida=unidad WHERE unidad_medida='Und' AND unidad!='' AND unidad!='Und'")
    except Exception:
        pass

    # bom: M4 espera cantidad_estandar
    add_col("bom", "cantidad_estandar", "REAL DEFAULT 0")
    try:
        c.execute("UPDATE bom SET cantidad_estandar=cantidad_requerida WHERE cantidad_estandar=0 AND cantidad_requerida>0")
    except Exception:
        pass

    # movimientos_inventario: M5 espera nota_orden
    add_col("movimientos_inventario", "nota_orden", "TEXT DEFAULT ''")


def _insert_defaults(c):
    # Empresa default
    c.execute("SELECT COUNT(*) FROM empresa")
    if c.fetchone()[0] == 0:
        c.execute("INSERT INTO empresa (razon_social) VALUES ('Mi Empresa C.A.')")

    # Parametros legales
    defaults = [
        ("sso_minimo", 9, "SSO Patronal (riesgo minimo)", "Reglam. SSO Art.192"),
        ("sso_medio", 10, "SSO Patronal (riesgo medio)", "Reglam. SSO Art.192"),
        ("sso_maximo", 11, "SSO Patronal (riesgo maximo)", "Reglam. SSO Art.192"),
        ("paro_forzoso", 2, "Paro Forzoso Patronal", "Ley SSS"),
        ("faov", 2, "FAOV Patronal", "G.O. 6.805 (2024)"),
        ("inces", 2, "INCES Patronal", "D-Ley INCES"),
        ("cepp", 9, "CEPP Proteccion Pensiones", "Ley Prot. Pens. 2024"),
        ("dias_utilidades_min", 30, "Dias Utilidades (minimo legal)", "LOTTT Art.131"),
        ("dias_bono_vac_base", 15, "Dias Bono Vacacional (1er ano)", "LOTTT Art.192"),
        ("margen_max_lopj", 30, "Margen Maximo Ganancia LOPJ", "LOPJ Art.32"),
        ("tope_sso_sm", 5, "Tope SSO (salarios minimos)", "Reglam. SSO"),
        ("horas_semanales", 44, "Horas laborables semanales", "LOTTT Art.173"),
        ("metodo_inventario", 1, "Metodo Valuacion (1=PEPS, 2=CPP)", "NIC 2 / LOPJ"),
    ]
    for clave, valor, desc, base in defaults:
        c.execute(
            "INSERT OR IGNORE INTO parametros_legales (clave, valor, descripcion, base_legal) VALUES (?,?,?,?)",
            (clave, valor, desc, base)
        )

    # Categorias de material default
    cats = [
        "Materia Prima",
        "Esmaltes y Recubrimientos",
        "Tintas",
        "Barnices",
        "Material de Empaque",
        "Insumos Varios",
    ]
    for cat in cats:
        c.execute("INSERT OR IGNORE INTO categorias_material (nombre) VALUES (?)", (cat,))

    # Periodo actual
    now = datetime.now()
    c.execute(
        "INSERT OR IGNORE INTO periodos (anio, mes) VALUES (?,?)",
        (now.year, now.month)
    )


def backup_db():
    os.makedirs(BACKUP_DIR, exist_ok=True)
    if os.path.exists(DB_PATH):
        ts = datetime.now().strftime("%Y%m%d_%H%M%S")
        dest = os.path.join(BACKUP_DIR, f"akemi_cost_{ts}.db")
        shutil.copy2(DB_PATH, dest)
        return dest
    return None


def get_param(clave):
    conn = get_connection()
    row = conn.execute(
        "SELECT valor FROM parametros_legales WHERE clave=?", (clave,)
    ).fetchone()
    conn.close()
    return row["valor"] if row else 0


def get_empresa():
    conn = get_connection()
    row = conn.execute("SELECT * FROM empresa LIMIT 1").fetchone()
    conn.close()
    return dict(row) if row else {}


def update_empresa(**kwargs):
    conn = get_connection()
    sets = ", ".join(f"{k}=?" for k in kwargs)
    vals = list(kwargs.values())
    conn.execute(f"UPDATE empresa SET {sets}, updated_at=datetime('now','localtime') WHERE id=1", vals)
    conn.commit()
    conn.close()
